/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.view;

import java.beans.PropertyEditor;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.MalformedURLException;
import java.net.URL;
import java.sql.Time;
import java.sql.Timestamp;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import javax.persistence.Id;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.jsp.JspContext;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspWriter;
import javax.servlet.jsp.PageContext;
import javax.servlet.jsp.el.ELException;
import javax.servlet.jsp.tagext.DynamicAttributes;
import javax.servlet.jsp.tagext.JspFragment;
import javax.servlet.jsp.tagext.SimpleTagSupport;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.LazyInitializationException;
import org.springframework.beans.InvalidPropertyException;
import org.springframework.beans.propertyeditors.CustomBooleanEditor;
import org.springframework.beans.propertyeditors.CustomDateEditor;
import org.springframework.beans.propertyeditors.CustomNumberEditor;
import org.springframework.orm.hibernate3.HibernateTemplate;

import br.com.linkcom.neo.bean.BeanDescriptor;
import br.com.linkcom.neo.bean.annotation.DescriptionProperty;
import br.com.linkcom.neo.bean.editors.CalendarEditor;
import br.com.linkcom.neo.bean.editors.HoraPropertyEditor;
import br.com.linkcom.neo.bean.editors.MoneyPropertyEditor;
import br.com.linkcom.neo.bean.editors.TimestampPropertyEditor;
import br.com.linkcom.neo.core.standard.Neo;
import br.com.linkcom.neo.exception.TagNotFoundException;
import br.com.linkcom.neo.persistence.QueryBuilder;
import br.com.linkcom.neo.types.Money;
import br.com.linkcom.neo.util.ReflectionCache;
import br.com.linkcom.neo.util.ReflectionCacheFactory;
import br.com.linkcom.neo.util.Util;

/**
 * @author rogelgarcia
 * @since 25/01/2006
 * @version 1.1
 */
public class BaseTag extends SimpleTagSupport implements DynamicAttributes {
	
	protected Log log = LogFactory.getLog(this.getClass());
	
	/**atributo que existe em todas as tags*/
	protected String id;
	
	protected Boolean printWhen;

	public Boolean getPrintWhen() {
		return printWhen;
	}

	public void setPrintWhen(Boolean printWhen) {
		this.printWhen = printWhen;
	}
	
	private String STACK_ATTRIBUTE_NAME = "TagStack";
	
	public String TAG_ATTRIBUTE = "tag";
	
	private Map<String, Object> dynamicAttributesMap = new HashMap<String, Object>();
	
	static private Map<String, String[]> templateCache = new HashMap<String, String[]>();
	static private Set<String> templates = new HashSet<String>();
	
	private BaseTag parent;
	//TODO JOGAR A INICIALIZACAO DOS PROPERTY EDITORS PARA OUTRO LUGAR
	protected Map<Class<?>, PropertyEditor> propertyEditors = new HashMap<Class<?>, PropertyEditor>();
	protected Map<Class<? extends Annotation>, InputListener<? extends Annotation>> inputListeners = new HashMap<Class<? extends Annotation>, InputListener<? extends Annotation>>();
	
	public void registerPropertyEditor(Class<?> class1, PropertyEditor editor){
		propertyEditors.put(class1, editor);
	}
	
	public void registerInputListener(InputListener<? extends Annotation> inputListener){
		inputListeners.put(inputListener.getAnnotationType(), inputListener);
	}
	
	@SuppressWarnings("unchecked")
	public <A extends Annotation> InputListener<A> getInputListener(A annotation){
		InputListener<A> inputListener = (InputListener<A>) inputListeners.get(annotation.annotationType());
		if(inputListener == null){
			return new InputListener(){
				public void onRender(InputTag input, Annotation annotation) {}
				public Class getAnnotationType() {return null;}
			};
		}
		return inputListener;
	}
	
	public String generateUniqueId(){
		Integer idsequence = (Integer) getRequest().getAttribute("IDSEQUENCE");
		if(idsequence == null){
			idsequence = 0;
		}
		getRequest().setAttribute("IDSEQUENCE", idsequence + 1);
		return ("" + ((char)(Math.random()*20+65))) + idsequence;
	}
	
	public BaseTag(){
    	final DecimalFormat numberFormat = new DecimalFormat("#.##############");
    	
    	final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("dd/MM/yyyy");
        final boolean allowEmpty = true;
    	
        registerPropertyEditor(Boolean.class, new CustomBooleanEditor(false));
        registerPropertyEditor(Short.class, new CustomNumberEditor(Short.class, false));
        registerPropertyEditor(Integer.class, new CustomNumberEditor(Integer.class, false));
        registerPropertyEditor(Long.class, new CustomNumberEditor(Long.class, false));
        registerPropertyEditor(BigInteger.class, new CustomNumberEditor(BigInteger.class, false));
        registerPropertyEditor(Float.class, new CustomNumberEditor(Float.class, numberFormat, false));
        registerPropertyEditor(Double.class, new CustomNumberEditor(Double.class, numberFormat, false));
        registerPropertyEditor(BigDecimal.class, new CustomNumberEditor(BigDecimal.class, numberFormat, false));
        
        registerPropertyEditor(Date.class, new CustomDateEditor(simpleDateFormat,allowEmpty));
        registerPropertyEditor(Calendar.class, new CalendarEditor(simpleDateFormat,allowEmpty));
        registerPropertyEditor(GregorianCalendar.class, new CalendarEditor(simpleDateFormat,allowEmpty));
        registerPropertyEditor(java.sql.Date.class, new CustomDateEditor(simpleDateFormat,allowEmpty));
        
        registerPropertyEditor(Time.class, new br.com.linkcom.neo.bean.editors.TimePropertyEditor());
        registerPropertyEditor(br.com.linkcom.neo.types.Hora.class, new HoraPropertyEditor());
        registerPropertyEditor(Timestamp.class, new TimestampPropertyEditor());
        
        registerPropertyEditor(Money.class, new MoneyPropertyEditor());

        registerInputListener(new MaxLengthInputListener());
        registerInputListener(new YearInputListener());
	}

	public BaseTag getParent() {
		return parent;
	}

	/**
	 * Faz o escape de aspas duplas
	 * @param opValue
	 * @return
	 */
	protected String escape(String opValue) {
		if(opValue==null) return null;
		
		return opValue.replaceAll("\\\\", "\\\\\\\\").replaceAll("\"", "\\\\\"");
	}
	
	protected String escapeSingleQuotes(String opValue) {
		if(opValue==null) return null;
		return opValue.replaceAll("\\\\", "\\\\\\\\").replaceAll("\'", "\\\\'");
	}
	
	@SuppressWarnings("unchecked")
	protected void pushAttribute(String name, Object value){
		getStack(name+"_stack").push(value);
		getRequest().setAttribute(name, value);
	}
	
	private Stack getStack(String string) {
		Stack stack = (Stack) getRequest().getAttribute(string);
		if(stack == null){
			stack = new Stack();
			getRequest().setAttribute(string, stack);
		}
		return stack;
	}

	@SuppressWarnings("unchecked")
	protected Object popAttribute(String name){
		Stack stack = getStack(name+"_stack");
		Object pop = stack.pop();
		if(!stack.isEmpty()){
			getRequest().setAttribute(name, stack.peek());
		} else {
			getRequest().setAttribute(name, null);
		}
		return pop;
	}
	
	/**
	 * Renderiza o corpo da tag
	 * @throws JspException
	 * @throws IOException
	 */
	protected final void doBody() throws JspException, IOException {
		if (getJspBody()!=null) {
			try {
				getJspBody().invoke(getOut());
			} catch (Exception e) {
				printException(e);
			}
		}
	}

	private void printException(Exception e) throws IOException {
		String extype = "";
		if(e.getClass().getName().startsWith("java.lang")){
			extype = e.getClass().getSimpleName() + ": ";
		}
		getOut().println("<font color=\"red\">" + extype +"<b>"+ e.getMessage() +"</b>"+ "</font>");
		Throwable cause = getNextException(e);
		while(cause != null){
			getOut().println("<font color=\"red\">" + cause.getMessage() + "</font>");
			cause = getNextException(cause);
		}
		e.printStackTrace();
	}
	
	private Throwable getNextException(Throwable e) {
		if(e == null){
			return null;
		}
		if(e instanceof ServletException){
			if(e.getCause() != null && e.getCause() != e){
				return e.getCause();
			} else {
				return ((ServletException)e).getRootCause();	
			}
		}
		return e.getCause();
	}

	/**
	 * Transforma o objeto em uma String cooreta para ser colocada em algum value
	 * Faz o IdStringStyle 
	 * @param value
	 * @return
	 */
	protected String getObjectValueToString(Object value, boolean includeDescription) {
		if(value == null) return "";
		PropertyEditor propertyEditor = propertyEditors.get(value.getClass());
		try {
			if(propertyEditor != null){
				propertyEditor.setValue(value);
				return propertyEditor.getAsText();
			} else if(hasId(value.getClass())){
				return Util.strings.toStringIdStyled(value, includeDescription);
			}else if(value instanceof Enum){
				return ((Enum)value).name();
			}
		
			return value.toString();
		} catch (LazyInitializationException e) {
			String id = "";
			try{
				id = Util.strings.toStringIdStyled(value, false);
			}catch(LazyInitializationException e2){
				
			}
			return value.getClass().getSimpleName()+" [No foi possvel fazer toString LazyInicializationException] "+id;
		} catch(InvalidPropertyException e1){
			
			Object object = Neo.getApplicationContext().getConfig().getProperties().get("autoLoadOnView");
			if(object == null || "true".equalsIgnoreCase(object.toString())){
				//UMA COISA MUITO FORADA.. TENTANDO LOADAR O OBJETO NA FORA
				try {
					log.warn("Perda de performance! Carregando objeto sob demanda "+value.getClass());
					if(value != null && value.getClass().getName().contains("$$") && e1.getCause().getCause() instanceof LazyInitializationException){
						value = new QueryBuilder(Neo.getObject(HibernateTemplate.class)).from(value.getClass().getSuperclass()).entity(value).unique();
						return getObjectValueToString(value, includeDescription);
					} else {
						throw e1;
					}
				} catch (NullPointerException e) {
					throw e1;
				}	
			}
			else {
				throw e1;
			}
		}
		
	}
	
	protected String getObjectValueToString(Object value) {
		return getObjectValueToString(value, false);
	}
	
	protected boolean hasId(Class<? extends Object> class1) {
		if(class1.getName().contains("$$")){
			//a classe enhanceada pelo Hibernate d pau ao fazer class.getMethods .. temos que pegar a classe superior nao enhanceada
			class1 = class1.getSuperclass();
		}
		ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
		while (class1 != null && !class1.equals(Object.class)) {
			Method[] methods = reflectionCache.getMethods(class1);
			for (Method method : methods) {
				if (reflectionCache.isAnnotationPresent(method, Id.class)) {
					return true;
				}
			}
			class1 = class1.getSuperclass();
		}
		return false;
	}


	protected String getObjectDescriptionToString(Object value) {
		if(value == null) return "";
		PropertyEditor propertyEditor = propertyEditors.get(value.getClass());
		try {
			if(propertyEditor != null){
				propertyEditor.setValue(value);
				return propertyEditor.getAsText();
			} else if(hasDescriptionProperty(value.getClass())){
				return Util.strings.toStringDescription(value);
			}
		
			return value.toString();
		} catch (LazyInitializationException e) {
			return value.getClass().getSimpleName()+" [No foi possvel fazer toString LazyInicializationException]";
		} catch(InvalidPropertyException e1){
			Object object = Neo.getApplicationContext().getConfig().getProperties().get("autoLoadOnView");
			if(object == null || "true".equalsIgnoreCase(object.toString())){
				//UMA COISA MUITO FORADA.. TENTANDO LOADAR O OBJETO NA FORA
				log.warn("Perda de performance! Carregando objeto sob demanda "+value.getClass());
				try {
					if(value != null && value.getClass().getName().contains("$$") && e1.getCause().getCause() instanceof LazyInitializationException){
						value = new QueryBuilder(Neo.getObject(HibernateTemplate.class)).from(value.getClass().getSuperclass()).entity(value).unique();
						return getObjectDescriptionToString(value);
					} else {
						throw e1;
					}
				} catch (NullPointerException e) {
					throw e1;
				}
			}
			else {
				if(e1.getCause() instanceof InvocationTargetException 
						&& ((InvocationTargetException)e1.getCause()).getTargetException() instanceof LazyInitializationException){
					throw new RuntimeException("No foi possvel fazer ObjectDescriptionToString de "+value.getClass().getSimpleName()+". (Talvez seja necessrio um fetch)", e1);
				} else {
					throw e1;	
				}
				
			}
		}
	}
	
	private boolean hasDescriptionProperty(Class<? extends Object> class1) {
		if(class1.getName().contains("$$")){
			//a classe enhanceada pelo Hibernate d pau ao fazer class.getMethods .. temos que pegar a classe superior nao enhanceada
			class1 = class1.getSuperclass();
		}
		ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
		while (!class1.equals(Object.class)) {
			Method[] methods = reflectionCache.getMethods(class1);
			for (Method method : methods) {
				if (reflectionCache.isAnnotationPresent(method, DescriptionProperty.class)) {
					return true;
				}
			}
			class1 = class1.getSuperclass();
		}
		return false;
	}

	public String getBody() throws JspException, IOException {
		ByteArrayOutputStream arrayOutputStream = new ByteArrayOutputStream();
		PrintWriter writer = new PrintWriter(arrayOutputStream);
		getJspBody().invoke(writer);
		writer.flush();
		return arrayOutputStream.toString();
	}
	
	protected JspWriter getOut() {
		return getPageContext().getOut();
	}
	
	@SuppressWarnings("unchecked")
	protected <E> E findParent(Class<E> tagClass) {
		return findParent(tagClass, false);
	}
	
	/**
	 * Retorna a primeira tag encontrada que for de alguma das classes passadas
	 * @param classes
	 * @return
	 */
	public BaseTag findFirst(Class<? extends BaseTag>... classes){
		List<BaseTag> tags = getTagsFromTopToThis();
		boolean found = false;
		for (Iterator iter = tags.iterator(); iter.hasNext();) {
			BaseTag element = (BaseTag) iter.next();
			if(element == this){
				found = true;
			}
			if(found){
				iter.remove();
			}
		}
		Collections.reverse(tags);
		List<Class<? extends BaseTag>> asList = Arrays.asList(classes);
		for (BaseTag tag : tags) {
			if(asList.contains(tag.getClass())){
				return tag;
			}
		}
		return null;
	}
	
	@SuppressWarnings("unchecked")
	protected BaseTag findFirst2(Class... classes){
		List<BaseTag> tags = getTagsFromThisToTop();
		tags.remove(0);
		List<Class> asList = Arrays.asList(classes);
		for (BaseTag tag : tags) {
			for (Class class1 : asList) {
				if(class1.isAssignableFrom(tag.getClass())){
					return tag;
				}
			}
		}
		return null;
	}
	
	/**
	 * Retorna a lista de tags ordenadas dessa tag(primeira) at a ultima(topo das tags)
	 * @return
	 */
	protected List<BaseTag> getTagsFromThisToTop() {
		List<BaseTag> tags = new ArrayList<BaseTag>();
		tags.addAll(getTagStack());
		Collections.reverse(tags);
		return tags;
	}
	
	protected List<BaseTag> getTagsFromTopToThis() {
		List<BaseTag> tags = new ArrayList<BaseTag>();
		tags.addAll(getTagStack());
		return tags;
	}

	@SuppressWarnings("unchecked")
	protected <E> E findParent(Class<E> tagClass, boolean throwExceptionIfNotFound) throws TagNotFoundException {
		Stack<BaseTag> tagStack = getTagStack();
		for (int i = tagStack.size()-2; i >= 0; i--) {
			BaseTag baseTag = tagStack.get(i);
			if(tagClass.equals(baseTag.getClass())){
				return (E)baseTag;
			}
		}
		if(throwExceptionIfNotFound){
			throw new TagNotFoundException("A tag "+this.getClass().getName()+" tentou procurar uma tag "+tagClass.getName()+" mas no encontrou. Provavelmente  obrigatrio a tag "+this.getClass().getName()+" estar aninhada a uma tag "+tagClass.getName());
		}
		return null;
	}
	
	/**
	 * Acha uma tag que for do tipo da classe passada ou subclasse. 
	 * @param <E>
	 * @param tagClass Classe da tag a ser encontrada. Pode ser passada uma tag pai ou interface que a tag possui.
	 * @param throwExceptionIfNotFound
	 * @return
	 * @throws TagNotFoundException
	 */
	@SuppressWarnings("unchecked")
	protected <E> E findParent2(Class<E> tagClass, boolean throwExceptionIfNotFound) throws TagNotFoundException {
		Stack<BaseTag> tagStack = getTagStack();
		for (int i = tagStack.size()-2; i >= 0; i--) {
			BaseTag baseTag = tagStack.get(i);
			if(tagClass.isAssignableFrom(baseTag.getClass())){
				return (E)baseTag;
			}
		}
		if(throwExceptionIfNotFound){
			throw new TagNotFoundException("A tag "+this.getClass().getName()+" tentou procurar uma tag "+tagClass.getName()+" mas no encontrou.");
		}
		return null;
	}

	@Override
	public final void doTag() throws JspException, IOException {
		if(Boolean.FALSE.equals(printWhen)){
			return;
		}
		boolean registeringDataGrid = getRequest().getAttribute(ColumnTag.REGISTERING_DATAGRID) != null;
		if(registeringDataGrid && !(this instanceof ColumnChildTag)){
			//se estiver registrando o datagrid no precisa renderizar nada
			return;
		}
		if (!getTagStack().isEmpty()) {
			parent = getTagStack().peek();
		}
		try{
			//coloca a tag no escopo
			colocaTagNaPilha();
	
			//verificar se est dentro de um panelGrid
			//panelGrid tem um comportamento especial para poder suportar tags dentro dele sem utilizar a tag Panel
			BaseTag parent = getParent();
			while (parent != null && parent instanceof LogicalTag) {
				parent = parent.getParent();
			}
			PanelGridTag panelGrid = parent instanceof PanelGridTag ? (PanelGridTag) parent : null;
			GetContentTag getContent = findParent(GetContentTag.class);
			if (getContent != null && getContent.getTag(this)) {
				doTagInGetContent(getContent);
			} else if (panelGrid != null && !(this instanceof PanelTag) && !(this instanceof LogicalTag)) {
				doTagInPanelGrid(panelGrid);
			} else {
				try {
					doComponent();
				} catch (JspException e) {
					throw e;
				} catch (Exception e) {
					try {
						printException(e);
					} catch (IOException e1) {
						throw new JspException(e);
					}
					//throw new JspException(e);
				}
			}
			//tira a tag do escopo
		} finally {
			tiraTagDaPilha();	
		}
		
		
	}
	
	protected void doTagInGetContent(GetContentTag getContent) throws JspException {
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
		PrintWriter printWriter = new PrintWriter(outputStream);
		getPageContext().pushBody(printWriter);
		try {
			doComponent();
		} catch (JspException e) {
			throw e;
		} catch (Exception e){
			throw new JspException(e);
		}
		getPageContext().popBody();
		printWriter.flush();
		String body = outputStream.toString();
		
		getContent.register(body);
	}

	private void doTagInPanelGrid(PanelGridTag panelGrid) throws JspException {
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
		PrintWriter printWriter = new PrintWriter(outputStream);
		getPageContext().pushBody(printWriter);
		try {
			doComponent();
		} catch (JspException e) {
			throw e;
		} catch (Exception e){
			throw new JspException(e);
		}
		getPageContext().popBody();
		printWriter.flush();
		String body = outputStream.toString();
		PanelRenderedBlock block = new PanelRenderedBlock();
		Map<String, Object> properties =new HashMap<String, Object>();
		addBasicPanelProperties(properties);
		addPanelProperties(properties);
		
		block.setBody(body);
		block.setProperties(properties);
		panelGrid.addBlock(block);
	}

	protected void addBasicPanelProperties(Map<String, Object> properties) {
		Set<String> keySet = dynamicAttributesMap.keySet();
		for (String string : keySet) {
			if(string.startsWith("panel")){
				properties.put(string.substring("panel".length()), dynamicAttributesMap.get(string));
			}
		}
	}

	protected void addPanelProperties(Map<String, Object> properties) {
		
	}

	protected void doComponent() throws Exception{
		
	}

	private void colocaTagNaPilha() {
		getTagStack().add(this);
	}

	private void tiraTagDaPilha() {
		getTagStack().pop();
	}

	protected Stack<BaseTag> getTagStack() {
		@SuppressWarnings("unchecked")
		Stack<BaseTag> stack = (Stack<BaseTag>) getRequest().getAttribute(STACK_ATTRIBUTE_NAME);
		if(stack == null){
			stack = new Stack<BaseTag>();
			getRequest().setAttribute(STACK_ATTRIBUTE_NAME, stack);
		}
		return stack;
	}

	protected HttpServletRequest getRequest() {
		return (HttpServletRequest)(getPageContext()).getRequest();
	}

	protected HttpServletResponse getResponse() {
		return (HttpServletResponse)getPageContext().getResponse();
	}

	protected ServletContext getServletContext() {
		return getPageContext().getServletContext();
	}

	
	protected PageContext getPageContext() {
		return (PageContext)getJspContext();
	}
	
	protected void includeTextTemplate() throws ServletException, IOException, ELException, JspException{
		String url = "/WEB-INF/classes/"+getTemplateName()+".jsp";
		verificarTemplate(null);
		includeTextTemplateFile(url);
	}

	protected void includeTextTemplate(String suffix) throws ServletException, IOException, ELException, JspException{
		String url = "/WEB-INF/classes/"+getTemplateName()+"-"+suffix+".jsp";
		verificarTemplate(suffix);
		includeTextTemplateFile(url);
	}
	
	private void verificarTemplate(String suffix) {
		String template = null;
		if(suffix == null){
			template = getTemplateName()+".jsp";
		} else {
			template = getTemplateName()+"-"+suffix+".jsp";
		}
		if(templates.contains(template)){
			return;
		}
		String url = "/WEB-INF/classes/"+template;
		try {
			URL resource = getServletContext().getResource(url);
			if(resource == null){
				//copiar do jar do neo
				
				//achar o jar do neo
				String neoJarPath = getNeoJarPath();
				
				byte[] bytesTemplate = getTemplate(template, neoJarPath);
				
				writeTemplate(template, bytesTemplate);
			} else {
				templates.add(template);
			}
		} catch (MalformedURLException e) {
			throw new RuntimeException(e);
		}
	}

	private void writeTemplate(String template, byte[] bytesTemplate) {
		if(bytesTemplate != null){
			String realPath = getServletContext().getRealPath("/WEB-INF/classes");
			int lastSeparator = Math.max(template.lastIndexOf('\\'), template.lastIndexOf('/'));
			String templatePath = template.substring(0, lastSeparator);
			
			File dir = new File(realPath + File.separator + templatePath);
			if(!dir.exists()){
				dir.mkdirs();
			}
			
			File arquivo = new File(realPath + File.separator + template);
			try {
				FileOutputStream outputStream = new FileOutputStream(arquivo);
				BufferedOutputStream bufferedOutputStream = new BufferedOutputStream(outputStream);
				for (int i = 0; i < bytesTemplate.length; i++) {
					bufferedOutputStream.write(bytesTemplate[i]);
				}
				bufferedOutputStream.flush();
				bufferedOutputStream.close();
			} catch (FileNotFoundException e) {
				log.warn("Arquivo no encontrado "+realPath + File.separator + template+"  "+ e.getMessage());
			} catch (IOException e) {
				log.warn("No foi possvel escrever o arquivo "+realPath + File.separator + template+"  "+ e.getMessage());
			}
		}
	}

	private byte[] getTemplate(String template, String neoJarPath) {
		byte[] bytesTemplate = null;
		
		//se tiver o jar do neo, procurar o arquivo l dentro
		if(neoJarPath != null){
			String root = getServletContext().getRealPath("/");
			//InputStream streamNeo = getServletContext().getResourceAsStream(neoJarPath);
			//ZipInputStream zip = new ZipInputStream(streamNeo);
			
			try {
				ZipFile zipFile = new ZipFile(root+neoJarPath);
				ZipEntry entry = zipFile.getEntry(template);
				InputStream inputStreamTemplate = zipFile.getInputStream(entry);
				BufferedInputStream bufferedInputStream = new BufferedInputStream(inputStreamTemplate);
				ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream((int) entry.getSize());
				BufferedOutputStream bufferedOutputStream = new BufferedOutputStream(byteArrayOutputStream);
				int data = -1;
				while((data = bufferedInputStream.read()) != -1){
					bufferedOutputStream.write(data);
				}
				bufferedInputStream.close();
				bufferedOutputStream.flush();
				bytesTemplate = byteArrayOutputStream.toByteArray();
				bufferedOutputStream.close();
			} catch (IOException e) {
				log.warn("No foi possvel copiar o template ("+neoJarPath+") do jar do neo para a aplicacao. " + e.getMessage());
			}
		}
		return bytesTemplate;
	}

	private String getNeoJarPath() {
		String neoJarPath = null;
		Set resourcePaths = getServletContext().getResourcePaths("/WEB-INF/lib");
		for (Object object : resourcePaths) {
			if(object.toString().contains("neo") && object.toString().endsWith(".jar")){
				neoJarPath = object.toString();
				break;
			}
		}
		return neoJarPath;
	}
	
	protected void includeJspTemplate() throws ServletException, IOException {
		String url = "/WEB-INF/classes/"+getTemplateName()+".jsp";
		verificarTemplate(null);
		includeJspTemplateFile(url);
	}

	protected String getTemplateName() {
		return this.getClass().getName().replaceAll("\\.", "/");
	}
	
	protected void includeJspTemplate(String suffix) throws ServletException, IOException {
		String url = "/WEB-INF/classes/"+getTemplateName()+"-"+suffix+".jsp";
		verificarTemplate(suffix);
		includeJspTemplateFile(url);
	}
	
	protected void includeTextTemplateFile(String template) throws ServletException, IOException, ELException, JspException{
		String[] text = templateCache.get(template);
		//cancela o cache
		text = null;
		if(text == null){
			InputStream resourceAsStream = getRequest().getSession().getServletContext().getResourceAsStream(template);
			if(resourceAsStream == null){
				throw new IllegalArgumentException("Template "+template+" no encontrado!");
			}
			BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(resourceAsStream));
			String line = null;
			String[] text2 = {"",""};
			boolean first = true;
			int index = 0;
			while((line = bufferedReader.readLine()) != null){
				if(line.contains("<dobody/>")) {
					String[] split = line.split("<dobody/>");
					if(split.length>0){
						text2[index] += split[0];	
					}
					if(split.length>1){
						text2[index+1] += split[1];	
					}
					index++;
					first = true;
					continue;
				}
				if(!first){
					text2[index] = text2[index] + "\n";
				} else {
					first = false;
				}
				text2[index] = text2[index] + line;
			}
			text = text2;
			templateCache.put(template, text);
		}
		Object last = getRequest().getAttribute(TAG_ATTRIBUTE);
		getRequest().setAttribute(TAG_ATTRIBUTE, this);
		
		evaluateAndPrint(text[0]);
		if(getJspBody()!= null){
			getJspBody().invoke(null);
			//tirei o println talvez d algum efeito colateral 31/08/2006 -> O println tava atrapalhando em determinadas tags
			//getOut().println();
		}
		if(!text[1].trim().equals("")){
			evaluateAndPrint(text[1]);
		}
		
		
		getRequest().setAttribute(TAG_ATTRIBUTE, last);
	}

	protected void evaluateAndPrint(String expression) throws ELException, IOException {
		Object evaluate = getPageContext().getExpressionEvaluator()
			.evaluate(expression, String.class, getPageContext().getVariableResolver(), null);
		getOut().print(evaluate);
	}
	
	protected Object evaluate(String expression) throws ELException {
		PageContext pageContext = getPageContext();
		return evaluate(expression, pageContext);
	}

	public static Object evaluate(String expression, JspContext pageContext) throws ELException {
		return pageContext.getExpressionEvaluator()
			.evaluate(expression, Object.class, pageContext.getVariableResolver(), null);
	}
	
	protected void evaluateAndPrintIncludeTag(String expression) throws ELException, IOException {
		Object last = getRequest().getAttribute(TAG_ATTRIBUTE);
		getRequest().setAttribute(TAG_ATTRIBUTE, this);
		Object evaluate = getPageContext().getExpressionEvaluator()
			.evaluate(expression, String.class, getPageContext().getVariableResolver(), null);
		getOut().print(evaluate);
		getRequest().setAttribute(TAG_ATTRIBUTE, last);
	}
	
	protected void includeJspTemplateFile(String template) throws ServletException, IOException{
		ByteArrayOutputStream arrayOutputStream = new ByteArrayOutputStream();
		PrintWriter writer = new PrintWriter(arrayOutputStream);
		//Object lastBody = getRequest().getAttribute(JSPFRAGMENT);
		//getRequest().setAttribute(JSPFRAGMENT, getJspBody());
		pushJspFragment(getRequest(), getJspBody());
		
		Object last = getRequest().getAttribute(TAG_ATTRIBUTE);
		getRequest().setAttribute(TAG_ATTRIBUTE, this);
		
		dispatchToTemplate(template, writer);
		writer.flush();
		getOut().write(arrayOutputStream.toString());
		//getOut().write(arrayOutputStream.toByteArray());
		
		getRequest().setAttribute(TAG_ATTRIBUTE, last);
		//getRequest().setAttribute(JSPFRAGMENT, lastBody);
		popJspFragment(getRequest());
	}
	
	public static JspFragment popJspFragment(HttpServletRequest request){
		List<JspFragment> jspFragmentStack = getJspFragmentStack(request);
		return jspFragmentStack.remove(jspFragmentStack.size() - 1);
	}
	
	public static void pushJspFragment(HttpServletRequest request, JspFragment jspFragment){
		List<JspFragment> jspFragmentStack = getJspFragmentStack(request);
		jspFragmentStack.add(jspFragment);
	}
	
	
	@SuppressWarnings("unchecked")
	public static List<JspFragment> getJspFragmentStack(HttpServletRequest request) {
		List<JspFragment> stack = (List<JspFragment>) request.getAttribute("JSPFRAGMENT");
		if(stack == null){
			stack = new ArrayList<JspFragment>();
			request.setAttribute("JSPFRAGMENT", stack);
		}
		return stack;
	}
	
	//Faz o dispatch para determinada url, coloca a saida no writer
	private void dispatchToTemplate(String template, PrintWriter writer) throws ServletException, IOException {
		WrappedWriterResponse response = new WrappedWriterResponse(getResponse(), writer);
		RequestDispatcher requestDispatcher = null;
		requestDispatcher = getRequest().getRequestDispatcher(template);
		requestDispatcher.include(getRequest(), response);
	}
	
	public Map<String, String[]> getTemplateCache() {
		return templateCache;
	}

	public void setDynamicAttribute(String uri, String localName, Object value) throws JspException {
		if(localName != null){
			localName = localName.toLowerCase();
		}
		dynamicAttributesMap.put(localName, value);
	}

	public Map<String, Object> getDynamicAttributesMap() {
		return dynamicAttributesMap;
	}

	@SuppressWarnings("unchecked")
	protected boolean isEntity(Class c) {
		return hasId(c);
	}

	/**
	 * Verifica a igualdade de dois objetos (analiza id)
	 * @param value1
	 * @param value2
	 * @return
	 */
	protected boolean areEqual(Object value1, Object value2) {
		boolean id = true;
		if(value1 == null){
			return false;
		} else {
			id = hasId(value1.getClass()) && id;
		}
		Class<? extends Object> class1 = value1.getClass();
		if(class1.getName().contains("$$")){
			class1 = class1.getSuperclass();
		}
		
		if(value2 == null){
			return false;
		} else {
			id = hasId(value2.getClass()) && id;
		}
		Class<? extends Object> class2 = value2.getClass();
		if(class2.getName().contains("$$")){
			class2 = class2.getSuperclass();
		}
		if(id){
			BeanDescriptor<Object> bd1 = Neo.getApplicationContext().getBeanDescriptor(value1);
			BeanDescriptor<Object> bd2 = Neo.getApplicationContext().getBeanDescriptor(value2);
			if(class1.equals(class2)){
				return bd1.getId().equals(bd2.getId());
			} else {
				boolean oneInstanceofOther = class1.isAssignableFrom(class2) || class2.isAssignableFrom(class1);
				if(oneInstanceofOther){
					//tentar verificar pelo id.. quando uma classe extender a outra
					return bd1.getId().equals(bd2.getId());
				}
			}
			return getObjectValueToString(value1, false).equals(getObjectValueToString(value2, false));
		}
		return value1.equals(value2);
	}

	public String getDynamicAttributesToString() {
		return getDynamicAttributesToString(dynamicAttributesMap);
	}
	
	public String getDynamicAttributesToString(Map<String, Object> dynamicAttributesMap) {
		StringBuilder builder = new StringBuilder(" ");
		Set<String> keySet = dynamicAttributesMap.keySet();
		for (String key : keySet) {
			builder.append(" ");
			builder.append(key);
			builder.append("=");
			builder.append("'");
			
			Object object = dynamicAttributesMap.get(key);
			if(object != null){
				if(object instanceof String && ((String)object).startsWith("ognl:")){
					object = ((String)object).substring(5);
					object = getOgnlValue((String)object);
				}
				object = object.toString();
			}
			builder.append(escapeSingleQuotes((String)object));
			builder.append("'");
		}
		String toString = builder.toString();
		return toString;
	}

	protected Object getOgnlValue(String expression) {
		WebContextMap contextMap = new WebContextMap(getRequest());			
		Object value = OgnlExpressionParser.parse(expression, contextMap);
		return value;
	}
	
	protected <E> E getOgnlValue(String expression, Class<E> expectedType) {
		WebContextMap contextMap = new WebContextMap(getRequest());			
		E value = OgnlExpressionParser.parse(expression, expectedType, contextMap);
		return value;
	}

	/**
	 * Atributo
	 * @return
	 */
	public String getId() {
		return id;
	}

	/**
	 * Atributo
	 * @return
	 */
	public void setId(String id) {
		this.id = id;
	}

	public void setDynamicAttributesMap(Map<String, Object> dynamicAttributesMap) {
		this.dynamicAttributesMap.putAll(dynamicAttributesMap);
	}

}

