/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.validation;

import java.lang.annotation.Annotation;
import java.sql.Date;
import java.sql.Time;
import java.util.HashMap;
import java.util.Map;

import br.com.linkcom.neo.core.config.ValidatorRegistry;
import br.com.linkcom.neo.types.Cep;
import br.com.linkcom.neo.types.Cnpj;
import br.com.linkcom.neo.types.Cpf;
import br.com.linkcom.neo.types.Hora;
import br.com.linkcom.neo.types.InscricaoEstadual;
import br.com.linkcom.neo.types.Telefone;
import br.com.linkcom.neo.validation.annotation.Email;
import br.com.linkcom.neo.validation.annotation.MaxLength;
import br.com.linkcom.neo.validation.annotation.MaxValue;
import br.com.linkcom.neo.validation.annotation.MinLength;
import br.com.linkcom.neo.validation.annotation.MinValue;
import br.com.linkcom.neo.validation.annotation.Required;
import br.com.linkcom.neo.validation.annotation.Year;
import br.com.linkcom.neo.validation.validators.ByteValidator;
import br.com.linkcom.neo.validation.validators.CepValidator;
import br.com.linkcom.neo.validation.validators.CnpjValidator;
import br.com.linkcom.neo.validation.validators.CpfValidator;
import br.com.linkcom.neo.validation.validators.DateValidator;
import br.com.linkcom.neo.validation.validators.EmailValidator;
import br.com.linkcom.neo.validation.validators.FloatValidator;
import br.com.linkcom.neo.validation.validators.InscricaoEstadualValidator;
import br.com.linkcom.neo.validation.validators.IntegerValidator;
import br.com.linkcom.neo.validation.validators.MaxLengthValidator;
import br.com.linkcom.neo.validation.validators.MaxValueValidator;
import br.com.linkcom.neo.validation.validators.MinLengthValidator;
import br.com.linkcom.neo.validation.validators.MinValueValidator;
import br.com.linkcom.neo.validation.validators.RequiredValidator;
import br.com.linkcom.neo.validation.validators.ShortValidator;
import br.com.linkcom.neo.validation.validators.TelefoneValidator;
import br.com.linkcom.neo.validation.validators.TimeValidator;
import br.com.linkcom.neo.validation.validators.YearValidator;


/**
 * Onde so registrados todos os validadores
 * @author rogelgarcia
 *
 */
public class ValidatorRegistryImpl implements ValidatorRegistry {

	protected Map<Class<? extends Annotation>, PropertyValidator> validators = new HashMap<Class<? extends Annotation>, PropertyValidator>();
	protected Map<Class<?>, PropertyValidator> typeValidators = new HashMap<Class<?>, PropertyValidator>();
	//TODO REGISTRAR OS TIPOS STRING
	protected Map<String, PropertyValidator> typeValidatorsString = new HashMap<String, PropertyValidator>();

	//public static final String validatorUtilities = "br/com/linkcom/neo/validation/validators/javascript/validateUtilities.js";
	
	protected ValidatorAnnotationExtractor annotationExtractor = new ValidatorAnnotationExtractorImpl(this);
	
	public ValidatorRegistryImpl(){
		register(Required.class, new RequiredValidator());
		register(Email.class, new EmailValidator());
		register(MaxLength.class, new MaxLengthValidator());
		register(MinLength.class, new MinLengthValidator());
		
		register(MaxValue.class, new MaxValueValidator());
		register(MinValue.class, new MinValueValidator());
		
		register(Year.class, new YearValidator());
		
		registerType(Float.class, new FloatValidator());
		registerType(Double.class, new FloatValidator());
		
		registerType(Integer.class, new IntegerValidator());
		registerType(Long.class, new IntegerValidator());
		registerType(Byte.class, new ByteValidator());
		registerType(Short.class, new ShortValidator());
		
		registerType(Date.class, new DateValidator());
		registerType(java.util.Date.class, new DateValidator());
		registerType(Time.class, new TimeValidator());
		registerType(Hora.class, new TimeValidator());
		
		registerType(Cpf.class, new CpfValidator());
		registerType(Cnpj.class, new CnpjValidator());
		registerType(InscricaoEstadual.class, new InscricaoEstadualValidator());
		registerType(Cep.class, new CepValidator());
		registerType(Telefone.class, new TelefoneValidator());
	}


	public PropertyValidator getPropertyValidator(Class<? extends Annotation> key) {
		return validators.get(key);
	}
	
	public PropertyValidator getTypeValidator(Class<?> key) {
		return typeValidators.get(key);
	}


	public PropertyValidator register(Class<? extends Annotation> key, PropertyValidator value) {
		return validators.put(key, value);
	}
	
	public PropertyValidator registerType(Class<?> key, PropertyValidator value) {
		return typeValidators.put(key, value);
	}
	
	public void clear() {
		validators.clear();
		typeValidators.clear();
	}


	public ValidatorAnnotationExtractor getExtractor() {
		return annotationExtractor;
	}


	public PropertyValidator getTypeValidator(String key) {
		return typeValidatorsString.get(key);
	}	
}
