/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import br.com.linkcom.neo.controller.ExtendedBeanWrapper;
import br.com.linkcom.neo.exception.NeoException;

public class CollectionsUtil {

	/**
	 * Verifica se determinada coleo possui uma propriedade com determinado valor. 
	 * Se a coleo for nula  retornado falso. <BR>
	 * Aceita nested properties
	 * @param collection
	 * @param property
	 * @param value
	 * @throws NullPointerException
	 * @return
	 */
	public boolean contains(Collection<?> collection, String property, Object value) throws NullPointerException{
		//TODO OTIMIZAR
		if(property == null){
			throw new NullPointerException("O parametro property no deve ser nulo");
		}
		if(collection == null){
			return false;
		}
		for (Object object : collection) {
			if(object != null){
				ExtendedBeanWrapper beanWrapper = new ExtendedBeanWrapper(object);
				try {
					Object objectPropertyValue = beanWrapper.getPropertyValue(property);
					if(objectPropertyValue == null){
						if(object == null){
							return true;
						}
					} else {
						if(objectPropertyValue.equals(object)){
							return true;
						}
					}
				} catch (IllegalArgumentException e) {
					throw new NeoException("Problema ao adquirir proprieade "+property+" do bean "+object, e);
				}
			}
		}
		return false;
	}
	
	/**
	 * Cria uma lista com uma propriedade de cada bean do collection fornecido
	 * 
	 * Ex.: Se tiver um collection de Pessoa e a pessoa tiver uma propriedade chamada nome
	 * Ser extraido o nome de cada pessoa e montado uma lista
	 * A lista no pode conter null 
	 * Todos os objetos da lista devem ser da mesma classe.
	 * Aceita nested properties
	 * @param collection Coleo de beans de onde deve ser extraido a propriedade
	 * @param property propriedade que deve ser extraida de cada bean
	 * @return Uma lista com os objetos de cada propriedade do bean
	 */
	@SuppressWarnings("unchecked")
	public static List<?> getListProperty(Collection<?> collection, String property){
		//TODO OTIMIZAR
		List list = new ArrayList();
		Iterator<?> iter = collection.iterator();
		if(iter.hasNext()){
			Object next = iter.next();
			ExtendedBeanWrapper beanWrapper = new ExtendedBeanWrapper(next);

			list.add(beanWrapper.getPropertyValue(property));

			while(iter.hasNext()){
				try {
					beanWrapper.setWrappedInstance(iter.next());
					list.add(beanWrapper.getPropertyValue(property));
				} catch (Exception e) {
					throw new RuntimeException(e);
				}
			}
		}
		return list;
	}
	
	/**
	 * Cria um HashSet com uma propriedade de cada bean do collection fornecido
	 * 
	 * Ex.: Se tiver um collection de Pessoa e a pessoa tiver uma propriedade chamada nome
	 * Ser extraido o nome de cada pessoa e montado uma lista
	 * A lista no pode conter null 
	 * Todos os objetos da lista devem ser da mesma classe.
	 * Aceita nested properties
	 * @param collection Coleo de beans de onde deve ser extraido a propriedade
	 * @param property propriedade que deve ser extraida de cada bean
	 * @return Uma lista com os objetos de cada propriedade do bean
	 */
	@SuppressWarnings("unchecked")
	public Set<?> getSetProperty(Collection<?> collection, String property){
		//TODO OTIMIZAR
		Set set = new HashSet();
		Iterator<?> iter = collection.iterator();
		if(iter.hasNext()){
			Object next = iter.next();
			ExtendedBeanWrapper beanWrapper = new ExtendedBeanWrapper(next);

			set.add(beanWrapper.getPropertyValue(property));

			while(iter.hasNext()){
				try {
					beanWrapper.setWrappedInstance(iter.next());
					set.add(beanWrapper.getPropertyValue(property));
				} catch (Exception e) {
					throw new RuntimeException(e);
				}
			}
		}
		return set;
	}
	
	/**
	 * Concatena todos os elementos de uma determinada collection e insere o token entre cada elemento
	 * @param collection Coleo a ser iteragida
	 * @param token String que deve ser usada entre cada elemento
	 * @return
	 */
	public static String concatenate(Collection<?> collection, String token){
		StringBuilder builder = new StringBuilder();
		for (Iterator<?> iter = collection.iterator(); iter.hasNext();) {
			Object o = iter.next();
			builder.append(o);
			if(iter.hasNext()){
				builder.append(token);
			}
		}
		return builder.toString();
	}
	
	public static String listAndConcatenate(Collection<?> collection, String property, String token){
		return concatenate(getListProperty(collection, property),token);
	}
	
	
	public static boolean contains(Collection<?> collection, Object elem) {
		for (Object object : collection) {
			if (object!=null) {
				if (elem !=null) {
					if (object.equals(elem)) {
						return true;
					}
				}
			}
			else {
				if (elem==null) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * Faz o toString de um para utilizando um separadorConjunto para separar os conjuntos de chave e valor e
	 * um separadorChaveValor que separa a chave do valor
	 */
	public String toString(Map<?, ?> mapa, String separadorConjunto, String separadorChaveValor) {
		StringBuilder builder = new StringBuilder();
		Set<?> keySet = mapa.keySet();
		for (Iterator iter = keySet.iterator(); iter.hasNext();) {
			Object object = iter.next();
			if(object != null){
				builder.append(object.toString());
			}
			
			builder.append(separadorChaveValor);
			
			Object value = mapa.get(object);
			if(value != null){
				builder.append(value.toString());
			}
			if(iter.hasNext()){
				builder.append(separadorConjunto);
			}
			
		}
		return builder.toString();
	}

	
}