/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.types;

import java.io.Serializable;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;

import org.apache.commons.lang.StringUtils;
import org.hibernate.HibernateException;
import org.hibernate.usertype.UserType;

public class Cnpj implements UserType, CadastroPessoa{

	private String value;
	
	/**
	 * 
	 * @deprecated Utilize o mtodo que recebe uma String, esse mtodo s existe por causa do hibernate
	 */
	@Deprecated
	public Cnpj(){
	}
	
	public Cnpj(String cnpj){
		if(cnpj == null) throw new NullPointerException();
		checkPattern(cnpj);
		String original = cnpj;
		cnpj = removeSymbols(cnpj);
		if(!cnpjValido(cnpj)){
			throw new IllegalArgumentException("No foi possvel converter \"" + original + "\" para um CNPJ vlido");
		}
		value = cnpj.trim().equals("")?null:cnpj;
	}
	
   /** Realiza a validao do CNPJ.
    *
    * @param   str_cnpj nmero de CNPJ a ser validado
    * @return  true se o CNPJ  vlido e false se no  vlido
    */
   public static boolean cnpjValido(String str_cnpj) {
	   
	   
		if (str_cnpj.length() != 15 && str_cnpj.length() != 14) {
			return false;
		}
		if(str_cnpj.length() == 15){
			str_cnpj = str_cnpj.substring(1,15);
		}
		
		int soma = 0, dig;
		String cnpj_calc = str_cnpj.substring(0, 12);

		if (str_cnpj.length() != 14)
			return false;

		char[] chr_cnpj = str_cnpj.toCharArray();

		/* Primeira parte */
		for (int i = 0; i < 4; i++)
			if (chr_cnpj[i] - 48 >= 0 && chr_cnpj[i] - 48 <= 9)
				soma += (chr_cnpj[i] - 48) * (6 - (i + 1));
		for (int i = 0; i < 8; i++)
			if (chr_cnpj[i + 4] - 48 >= 0 && chr_cnpj[i + 4] - 48 <= 9)
				soma += (chr_cnpj[i + 4] - 48) * (10 - (i + 1));
		dig = 11 - (soma % 11);

		cnpj_calc += (dig == 10 || dig == 11) ? "0" : Integer.toString(dig);

		/* Segunda parte */
		soma = 0;
		for (int i = 0; i < 5; i++)
			if (chr_cnpj[i] - 48 >= 0 && chr_cnpj[i] - 48 <= 9)
				soma += (chr_cnpj[i] - 48) * (7 - (i + 1));
		for (int i = 0; i < 8; i++)
			if (chr_cnpj[i + 5] - 48 >= 0 && chr_cnpj[i + 5] - 48 <= 9)
				soma += (chr_cnpj[i + 5] - 48) * (10 - (i + 1));
		dig = 11 - (soma % 11);
		cnpj_calc += (dig == 10 || dig == 11) ? "0" : Integer.toString(dig);

		return str_cnpj.equals(cnpj_calc);
	}
	
	private void checkPattern(String value) throws IllegalArgumentException {
		if(!value.trim().equals("") && !value.matches("\\d{2,3}\\.?\\d{3}\\.?\\d{3}/?\\d{4}-?\\d{2}")){
			throw new IllegalArgumentException("No foi possvel converter \"" + value + "\" para um CNPJ vlido");
		}
	}

	public String getValue() {
		return value;
	}

	@Override
	protected Cnpj clone() throws CloneNotSupportedException {
		return new Cnpj(value);
	}

	@Override
	public boolean equals(Object obj) {
		if(!(obj instanceof Cnpj)){
			return false;
		}
		if(this.value == null && ((Cnpj)obj).value == null){
			return true;
		} else if(this.value == null || ((Cnpj)obj).value == null){
			return false;
		}
		return this.value.equals(((Cnpj)obj).value);
	}

	@Override
	public int hashCode() {
		if(value == null) return super.hashCode();
		return value.hashCode();
	}

	@Override
	public String toString() {
		if(StringUtils.isEmpty(value)){
			return "";
		} else {
			try {
				StringBuilder builder = new StringBuilder(value);

				builder.insert(12, '-');
				builder.insert(8, '/');
				builder.insert(5, '.');
				builder.insert(2, '.');
				
				return builder.toString();
			} catch (IndexOutOfBoundsException e) {
				//System.out.println("\n************************\nCnpj invlido: "+value);
				return value;
			}
		}
	}

	public int[] sqlTypes() {
		return new int[]{Types.VARCHAR};
	}

	public Class returnedClass() {
		return Cnpj.class;
	}

	public boolean equals(Object x, Object y) throws HibernateException {
		if((x == null || ((Cnpj)x).value == null) && (y == null || ((Cnpj)y).value == null)){
			return true;
		} else if ((x == null && ((Cnpj)x).value == null)|| (y == null && ((Cnpj)y).value == null)){
			return false;
		}
		return x.equals(y);
	}

	public int hashCode(Object x) throws HibernateException {
		return x.hashCode();
	}

	public Object nullSafeGet(ResultSet rs, String[] names, Object owner) throws HibernateException, SQLException {
		String value = rs.getString(names[0]);
		if(value == null){
			return new Cnpj();
		}
		Cnpj cnpj = new Cnpj();
		cnpj.value = value;
		return cnpj;
	}

	public void nullSafeSet(PreparedStatement st, Object value, int index) throws HibernateException, SQLException {
		if(value instanceof Cnpj){
			String value2 = ((Cnpj)value).getValue();
			if(StringUtils.isEmpty(value2)){
				st.setNull(index, Types.VARCHAR);
			} else {
				st.setString(index, removeSymbols(value2));	
			}
				
		} else {
			st.setNull(index, Types.VARCHAR);
		}
	}

	private String removeSymbols(String value2) {
		return value2.replace(".", "").replace("-", "").replace("/", "");
	}

	public Object deepCopy(Object value) throws HibernateException {
		return value;
	}

	public boolean isMutable() {
		return false;
	}

	public Serializable disassemble(Object value) throws HibernateException {
		return ((Cnpj)value).getValue();
	}

	public Object assemble(Serializable cached, Object owner) throws HibernateException {
		return new Cnpj((String)cached);
	}

	public Object replace(Object original, Object target, Object owner) throws HibernateException {
		return original;
	}
	
	public static void main(String[] args) {
		String c = "006.213.117/0001-30";
		new Cnpj(c);
		System.out.println(Cnpj.cnpjValido(c));
	}
}
