/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.persistence;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.EntityMode;
import org.hibernate.HibernateException;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.metadata.ClassMetadata;
import org.hibernate.metadata.CollectionMetadata;
import org.hibernate.persister.entity.AbstractEntityPersister;
import org.hibernate.type.CollectionType;
import org.hibernate.type.ManyToOneType;
import org.hibernate.type.Type;
import org.springframework.orm.hibernate3.HibernateCallback;
import org.springframework.orm.hibernate3.HibernateTemplate;
import org.springframework.orm.hibernate3.SessionFactoryUtils;

import br.com.linkcom.neo.util.ReflectionCache;
import br.com.linkcom.neo.util.ReflectionCacheFactory;

/**
 * Facilitador para salvar e atualizar entidades
 * @author rogelgarcia
 *
 */
public class SaveOrUpdateStrategy {
	
	private Object entity;
	private HibernateTemplate hibernateTemplate;
	//private Session session;
	private List<HibernateCallback> firstCallbacks = new ArrayList<HibernateCallback>();
	private List<HibernateCallback> callbacks = new ArrayList<HibernateCallback>();

	private List<HibernateCallback> attachments = new ArrayList<HibernateCallback>();
	
	/**
	 * Cria um Save or Update managed
	 * 
	 * @param hibernateTemplate
	 * @param entity Entidade que ser salva
	 */
	public SaveOrUpdateStrategy(HibernateTemplate hibernateTemplate, Object entity){
		this.hibernateTemplate = hibernateTemplate;
		this.entity = entity;
	}
	
	/**
	 * Retorna a entidade que est sendo salva
	 * @return
	 */
	public Object getEntity() {
		return entity;
	}

	/**
	 * 
	 * @param entity Entidade que ser salva
	 * @return
	 */
	public SaveOrUpdateStrategy setEntity(Object entity) {
		this.entity = entity;
		return this;
	}

	/**
	 * Seta para cada objeto da coleo em path a entidade pai
	 * Path  o nome da propriedade, que deve ser um collection, onde encontram-se beans que tem referencia para
	 * a entidade que est sendo salva
	 * parentProperty  o nome da propriedade em cada bean da coleo que faz referencia a entidade sendo salva
	 * ex.:
	 * 
	 * A classe Pessoa tem uma referencia para municipio
	 * A classe Municipio tem um conjunto de referencias para Pessoa
	 * 
	 * Voc ir salvar municipio mas precisa setar em todas as pessoas o municipio em que elas moram
	 * 
	 * Criando um saveOrUpdateStrategy voce diz que a entidade  municipio
	 * e fala que todas em todas as pessoas tem que ser executado pessoa.setMunicipio
	 * 
	 * setParent("pessoas","municipio")
	 * 
	 * pessoas  o conjunto que existe em municipio
	 * municipio  a propriedade em pessoa que faz referencia a municipio
	 * 
	 * @param path
	 * @param parentProperty
	 * @return
	 */
	public SaveOrUpdateStrategy setParent(String path, String parentProperty){
		try {
			ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
			Method method = reflectionCache.getMethod(entity.getClass(), "get"+StringUtils.capitalize(path));
			Collection collection = (Collection)method.invoke(entity);
			Method parentSetMethod = null;
			if (collection != null) {
				for (Object object : collection) {
					if (parentSetMethod == null) {
						Method[] methods = reflectionCache.getMethods(object.getClass());
						for (Method setMethod : methods) {
							if (setMethod.getName().equals("set" + StringUtils.capitalize(parentProperty))) {
								Class<?>[] parameterTypes = setMethod.getParameterTypes();
								if (parameterTypes.length == 1 && parameterTypes[0].isAssignableFrom(entity.getClass())) {
									if (parentSetMethod != null) {
										throw new RuntimeException("No  possvel determinar mtodo setter para " + entity.getClass().getName() + " na classe " + object.getClass().getName()
												+ ". Existe mais de um setter");
									}
									//TODO MELHORAR.. TENTAR ACHAR O SETTER ESPECFICO E NO UM NA HIERARQUIA DA CLASSE
									parentSetMethod = setMethod;
								}
							}
						}
						//parentSetMethod = object.getClass().getMethod("set"+StringUtils.capitalize(parentProperty), entity.getClass());
						if (parentSetMethod == null) {
							throw new RuntimeException("No  possvel determinar mtodo setter para " + entity.getClass().getName() + " '"+"set" + StringUtils.capitalize(parentProperty)+"()' na classe " + object.getClass().getName()
									+ ". No existe nenhum setter");
						}
					}
					parentSetMethod.invoke(object, entity);
				}
			}
			return this;
		} catch (Exception e) {
			throw new RuntimeException("Problema ao configurar a propriedade pai", e);
		}
	}
	
	/**
	 * Salva a entidade.
	 * @return
	 */
	public SaveOrUpdateStrategy saveEntity(){
		HibernateCallback callback = new HibernateCallback(){
			
			public Object doInHibernate(Session session) throws HibernateException, SQLException {
				session.clear();
				session.saveOrUpdate(entity);
				return null;
			}
			
		};
		callbacks.add(callback);
		return this;
	}
	
	/**
	 * Excecuta as operaes pedidas at o momento
	 * @return
	 */
	public SaveOrUpdateStrategy flush(){
		return flush(false);
	}

	/**
	 * Excecuta as operaes pedidas at o momento, mas d preferencia aos comandos que devem ser executados primeiro
	 * Um exemplo de comando que  executado primeiro sao as deleoes dos detalhes
	 * @return
	 */
	private SaveOrUpdateStrategy flush(boolean insertFirst) {
		HibernateCallback callback = new HibernateCallback(){
			
			public Object doInHibernate(Session session) throws HibernateException, SQLException {
				session.flush();
				return null;
			}
			
		};
		if (insertFirst) {
			firstCallbacks.add(callback);
		} else {
			callbacks.add(callback);
		}
		return this;
	}
	
	/**
	 * Salva todos os objetos da coleao determinado por path
	 * path  o nome da propriedade na entidade que possui a coleao a ser salva
	 * @param path
	 * @return
	 */
	public SaveOrUpdateStrategy saveCollection(String path){
		try {
			ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
			Method method = reflectionCache.getMethod(entity.getClass(), "get"+StringUtils.capitalize(path));
			final Collection<?> collection = (Collection)method.invoke(entity);
			callbacks.add(new HibernateCallback(){
				public Object doInHibernate(Session session) throws HibernateException, SQLException {
					if(collection == null){
						//se a colecao  nula nao devemos salv-la
						return null;
					}
					for (Iterator<?> it = collection.iterator(); it.hasNext();) {
						Object next = it.next();
						session.saveOrUpdate(next);
					}
					return null;
				}
			});
			return this;
		} catch (Exception e) {
			throw new RuntimeException(e);
		} 
	}
	
	/**
	 * Deleta do banco as propriedades que nao foram encontrados no objeto
	 * 
	 * @param path Nome da propriedade com a coleo
	 * @param parentProperty Nome da propriedade nas classes filhas que fazem referencia a entidade
	 * @param itemClass Classe dos itens da colecao indicada por path
	 * @return
	 */
	public SaveOrUpdateStrategy deleteNotInEntity(String path, final String parentProperty, final Class itemClass){
		return deleteNotInEntity(path, parentProperty, itemClass, false);
	}

	public SaveOrUpdateStrategy deleteNotInEntity(String path) {
		SessionFactory sessionFactory = hibernateTemplate.getSessionFactory();

		Class itemClass = getItemClass(path, sessionFactory);

		String parentProperty = getParentProperty(sessionFactory, itemClass);
		
		return deleteNotInEntity(path, parentProperty, itemClass);
	}
	
	/**
	 * Deleta do banco as propriedades que nao foram encontrados no objeto
	 * 
	 * @param path Nome da propriedade com a coleo
	 * @param parentProperty Nome da propriedade nas classes filhas que fazem referencia a entidade
	 * @param itemClass Classe dos itens da colecao indicada por path
	 * @param insertFirst Se for true d preferencia a esse comando na hora de executar
	 * @return
	 */
	private SaveOrUpdateStrategy deleteNotInEntity(String path, final String parentProperty, final Class itemClass, boolean insertFirst) {
		try {
			Serializable entityid = hibernateTemplate.getSessionFactory().getClassMetadata(entity.getClass()).getIdentifier(entity, EntityMode.POJO);
			if(entityid == null){
				// se a entidade ainda nao est no banco nao existe nenhuma entidade no banco que nao esteje 
				// na propriedade
				return this;
			}
			
			ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
			Method method = reflectionCache.getMethod(entity.getClass(), "get"+StringUtils.capitalize(path));
			final Collection<?> collection = (Collection)method.invoke(entity);
			//Query deleteQuery = null;
			HibernateCallback deleteCallback;
			if(collection == null || collection.size() == 0){
				final String deleteQueryString = new StringBuilder()
				.append("delete ")
				.append(itemClass.getName())
				.append(" where ")
				.append(parentProperty)
				.append(" = :")
				.append(parentProperty)
				.toString();
				final String _parentProperty = parentProperty;
				deleteCallback = new HibernateCallback(){
					public Object doInHibernate(Session session) throws HibernateException {
						Query queryObject = session.createQuery(deleteQueryString);
						queryObject.setEntity(_parentProperty, entity);
						queryObject.executeUpdate();
						return null;
					}
				};
				
			} else {
				final ClassMetadata metadata = hibernateTemplate.getSessionFactory().getClassMetadata(itemClass);
				//final List<Serializable> ids = new ArrayList<Serializable>();
				final List toDelete = findItensToDelete(parentProperty, itemClass, collection, metadata);
				if(toDelete.size() == 0){
					return this;
				}
				deleteCallback = new HibernateCallback(){
					public Object doInHibernate(Session session) throws HibernateException {
						//session = session.getSession(EntityMode.POJO);
						//Transaction transaction = session.beginTransaction();
						
						for (Object object : toDelete) {
							session.delete(object);
							//session.flush();
						}
						//session.flush();
						//transaction.commit();
						return null;
					}
				};

			}
			if (insertFirst) {
				firstCallbacks.add(deleteCallback);
			} else {
				callbacks.add(deleteCallback);
			}
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return this;
	}

	private List findItensToDelete(final String parentProperty, final Class itemClass, final Collection<?> collection, final ClassMetadata metadata) {
		final List toDelete = hibernateTemplate.executeFind(new HibernateCallback(){
		
				public Object doInHibernate(Session session) throws HibernateException, SQLException {
					
					
					// remover dessa lista os objetos transientes
					Collection<Object> itens = new ArrayList<Object>(collection);
					for (Iterator<Object> iter = itens.iterator(); iter.hasNext();) {
						Object entity = iter.next();
						Serializable id = metadata.getIdentifier(entity, EntityMode.POJO);
						if(id == null){
							iter.remove();
						}
					}
					if(itens.size() == 0){
						// se nao tiver itens remover os que estiverem no banco
						final String findQueryString = new StringBuilder()
						.append("from ")
						.append(itemClass.getName())
						.append(" ")
						.append(StringUtils.uncapitalize(itemClass.getSimpleName()))
						.append(" where ")
						.append(StringUtils.uncapitalize(itemClass.getSimpleName()))
						.append('.')
						.append(parentProperty)
						.append(" = :")
						.append(entity.getClass().getSimpleName())
						.toString();
						
						Query q = session.createQuery(findQueryString);
						q.setEntity(entity.getClass().getSimpleName(), entity);
						return q.list();
					} else {
						// remover apenas os que nao estao na lista
						final String findQueryString = new StringBuilder()
						.append("from ")
						.append(itemClass.getName())
						.append(" ")
						.append(StringUtils.uncapitalize(itemClass.getSimpleName()))
						.append(" where ")
						.append(StringUtils.uncapitalize(itemClass.getSimpleName()))
						.append(" not in (:collection)  and ")
						.append(StringUtils.uncapitalize(itemClass.getSimpleName()))
						.append('.')
						.append(parentProperty)
						.append(" = :")
						.append(entity.getClass().getSimpleName())
						.toString();
						
						Query q = session.createQuery(findQueryString);
						q.setParameterList("collection", itens);
						q.setEntity(entity.getClass().getSimpleName(), entity);
						return q.list();
					}
					
				}

		});
		return toDelete;
	}
	
	/**
	 * Salva cada objeto novo na coleao indicada por path no banco
	 * Seta a propriedade pai de cada item da colecao para a entidade sendo salva
	 * Deleta do banco os itens nao encontrados na coleao
	 * @param path
	 * @param parentProperty
	 * @param itemClass
	 * @return
	 */
	public SaveOrUpdateStrategy saveOrUpdateManaged(String path, String parentProperty, Class itemClass){
		executeManagedSaving(path, itemClass, parentProperty);
		return this;
	}
	
	/**
	 * Deleta do banco os itens nao encontrados na coleao e d um flush na conexao
	 * 
	 * Salva cada objeto novo na coleao indicada por path no banco
	 * Seta a propriedade pai de cada item da colecao para a entidade sendo salva
	 * 
	 * @param path
	 * @return
	 */
	public SaveOrUpdateStrategy saveOrUpdateManagedDeleteFirst(String path){
		try {
			SessionFactory sessionFactory = hibernateTemplate.getSessionFactory();

			Class itemClass = getItemClass(path, sessionFactory);

			String parentProperty = getParentProperty(sessionFactory, itemClass);

			executeManagedSavingDeleteFirst(path, itemClass, parentProperty);
			
		} catch (Exception e) {
			throw new RuntimeException("No foi possvel usar o saveOrUpdateManaged(String) para "+entity.getClass().getName()+"! Possveis causas: " +
					"A os itens do collection no possuem referencia para o pai, O path estava incorreto. O path leva a uma coleo que no tem classe persistente", e);
		}
		return this;
	}

	/**
	 * Deleta do banco os itens nao encontrados na coleao e d um flush na conexao
	 * 
	 * Salva cada objeto novo na coleao indicada por path no banco
	 * Seta a propriedade pai de cada item da colecao para a entidade sendo salva
	 * 
	 * @param path
	 * @param parentProperty
	 * @param itemClass
	 * @return
	 */
	private void executeManagedSavingDeleteFirst(String path, Class itemClass, String parentProperty) {
		deleteNotInEntity(path, parentProperty, itemClass, true);
		flush(true);
		setParent(path, parentProperty);
		saveCollection(path);
	}
	
	/**
	 * Deleta do banco os itens nao encontrados na coleao e d um flush na conexao
	 * 
	 * Salva cada objeto novo na coleao indicada por path no banco
	 * Seta a propriedade pai de cada item da colecao para a entidade sendo salva
	 * 
	 * @param path
	 * @return
	 */
	public SaveOrUpdateStrategy saveOrUpdateManaged(String path){
		return saveOrUpdateManagedDeleteFirst(path);
	}

	/**
	 * 
	 * Salva cada objeto novo na coleao indicada por path no banco
	 * Seta a propriedade pai de cada item da colecao para a entidade sendo salva
	 * Deleta do banco os itens nao encontrados na coleao
	 * 
	 * @param path
	 * @return
	 */
	public SaveOrUpdateStrategy saveOrUpdateManagedNormal(String path){
		try {
			SessionFactory sessionFactory = hibernateTemplate.getSessionFactory();

			Class itemClass = getItemClass(path, sessionFactory);

			String parentProperty = getParentProperty(sessionFactory, itemClass);

			executeManagedSaving(path, itemClass, parentProperty);
			
		} catch (Exception e) {
			throw new RuntimeException("No foi possvel usar o saveOrUpdateManaged(String) para "+entity.getClass().getName()+"! Possveis causas: " +
					"A os itens do collection no possuem referencia para o pai, O path estava incorreto. O path leva a uma coleo que no tem classe persistente", e);
		}
		return this;
	}
	@SuppressWarnings("unchecked")
	private String getParentProperty(SessionFactory sessionFactory, Class itemClass) {
		AbstractEntityPersister persister = (AbstractEntityPersister) sessionFactory.getClassMetadata(itemClass);

		int i = 0;
		List<Integer> propriedadesPai = new ArrayList<Integer>();
		for (Type ptype : persister.getPropertyTypes()) {
			if (ptype instanceof ManyToOneType) {
				ManyToOneType manyToOneType = ((ManyToOneType) ptype);
				Class returnedClass = manyToOneType.getReturnedClass();
				if (returnedClass.isAssignableFrom(entity.getClass())) {
					manyToOneType.getName();
					propriedadesPai.add(i);
					//break;
				}
			}
			i++;
		}
		if(propriedadesPai.size() > 1){
			throw new RuntimeException("No  possvel determinar a classe pai para a propriedade. " +
							"Existem "+propriedadesPai.size()+" referencias da classe "+itemClass.getSimpleName()+" para a classe "+entity.getClass().getSimpleName());
		}
		if(propriedadesPai.size() == 0){
			throw new RuntimeException("No  possvel determinar a classe pai para a propriedade. " +
							"No existem referencias da classe "+itemClass.getSimpleName()+" para a classe "+entity.getClass().getSimpleName());
		}
		String parentProperty = persister.getPropertyNames()[propriedadesPai.get(0)];
		return parentProperty;
	}

	private Class getItemClass(String path, SessionFactory sessionFactory) {
		AbstractEntityPersister metadata = (AbstractEntityPersister) sessionFactory.getClassMetadata(entity.getClass());
		CollectionType type = (CollectionType) metadata.getPropertyType(path);
		String role = type.getRole();
		CollectionMetadata collectionMetadata = sessionFactory.getCollectionMetadata(role);
		Type collectionElementType = collectionMetadata.getElementType();
		Class itemClass = collectionElementType.getReturnedClass();
		return itemClass;
	}

	private void executeManagedSaving(String path, Class itemClass, String parentProperty) {
		setParent(path, parentProperty);
		deleteNotInEntity(path, parentProperty, itemClass);
		saveCollection(path);
	}
	
	/**
	 * Excecuta as os comandos desse saveOrUpdateStrategy<BR>
	 * E dos saveOrUpdateStrategy anexados
	 *
	 */
	public void execute(){
		
		Session session = SessionFactoryUtils.getSession(hibernateTemplate.getSessionFactory(), hibernateTemplate.getEntityInterceptor(), hibernateTemplate.getJdbcExceptionTranslator());
		Transaction transaction = session.getTransaction();
		boolean intransaction = true;
		if(!transaction.isActive()){
			transaction = session.beginTransaction();	
			intransaction = false;
		}
		 
		try {
			
			hibernateTemplate.execute(new HibernateCallback() {

				public Object doInHibernate(Session session)
						throws HibernateException, SQLException {

					List<HibernateCallback> callbacks = getCallbacks();

					for (HibernateCallback callback : callbacks) {
						callback.doInHibernate(session);
					}
					return null;
				}

			});
			if (!intransaction) {
				transaction.commit();
			}
		} catch (RuntimeException e) {
			if (!intransaction) {
				transaction.rollback();
			}
			throw e;
		} catch (Exception e){
			if (!intransaction) {
				transaction.rollback();
			}
			throw new RuntimeException(e);
		} finally {
			SessionFactoryUtils.releaseSession(session, hibernateTemplate.getSessionFactory());
				
		}
		
	}
	
	/**
	 * Utilizar esse mtodo se existir uma transao sendo executada
	 * @deprecated O mtodo execute j sabe se existe uma transao ativa.
	 */
	@Deprecated
	public void executeInTransaction(){
		hibernateTemplate.execute(new HibernateCallback(){

			public Object doInHibernate(Session session) throws HibernateException, SQLException {
				List<HibernateCallback> callbacks = getCallbacks();
				
				for (HibernateCallback callback : callbacks) {
					callback.doInHibernate(session);
				}
				return null;
			}
			
		}
		);
	}
	
	/**
	 * Excecuta as os comandos desse saveOrUpdateStrategy<BR>
	 * E dos saveOrUpdateStrategy anexados
	 * Recebe a sesso onde devem ser executados os comandos. O controle de transao
	 *  de responsabilidade de quem utilizar esse cdigo
	 *//*
	public void execute(Session session){
		
		List<HibernateCallback> callbacks = getCallbacks();
		
		for (HibernateCallback callback : callbacks) {
			try {
				callback.doInHibernate(session);
			} catch (SQLException e) {
				throw new RuntimeException(e);
			}
		}
		
	}
	*/
	/**
	 * Anexa um outro save ou update a esse.<BR>
	 * Depois que esse saveOrUpdate tiver concluido suas tarefas ele executar
	 * as tarefas dos saveOrUpdates anexados.<BR>
	 * Tanto as tarefas desse strategy quanto do anexado sero executadas na mesma transao.<BR>
	 * As tarefas sero verificadas na chamada desse mtodo, ento, se forem adicionadas tarefas
	 * depois da chamada desse mtodo, para o strategy fornecido, elas NO sero excecutadas 
	 * IMPORTANTE: NO CHAME O MTODO EXCECUTE NOS SAVEORUPDATESTRATEGYS ANEXADOS<BR>
	 *  
	 * @param strategy
	 */
	public SaveOrUpdateStrategy attach(SaveOrUpdateStrategy strategy){
		if(strategy == null) throw new NullPointerException("SaveOrUpdateStrategy null");
		attachments.addAll(strategy.getCallbacks());
		return this;
	}
	
	/**
	 * Anexa uma tarefa a essa estratgia.<BR>
	 * Depois que esse saveOrUpdate tiver concluido suas tarefas ele executar
	 * as tarefas anexadas.<BR>
	 * Tanto as tarefas desse strategy quanto o anexo sero executadas na mesma transao. 
	 * No  necessrio criar um contexto transacional na tarefa anexada<BR>
	 * @param callback
	 * @return
	 */
	public SaveOrUpdateStrategy attach(HibernateCallback callback){
		if(callback == null) throw new NullPointerException("HibernateCallback null");
		attachments.add(callback);
		return this;
	}
	
	/**
	 * Arruma os callbacks na ordem que devem ser chamados
	 * Inclui os callbacks dos saveorupdatestrategys anexados
	 * @return
	 */
	private List<HibernateCallback> getCallbacks(){
		List<HibernateCallback> callbacks = new ArrayList<HibernateCallback>();
		callbacks.addAll(this.firstCallbacks);
		callbacks.addAll(this.callbacks);
		callbacks.addAll(attachments);
		return callbacks;
	}
}
