/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.persistence;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.EntityMode;
import org.hibernate.Hibernate;
import org.hibernate.HibernateException;
import org.hibernate.Query;
import org.hibernate.Session;
import org.springframework.orm.hibernate3.HibernateCallback;
import org.springframework.orm.hibernate3.HibernateTemplate;
import org.springframework.orm.hibernate3.SessionFactoryUtils;

import br.com.linkcom.neo.controller.crud.FiltroListagem;
import br.com.linkcom.neo.core.standard.Neo;
import br.com.linkcom.neo.exception.UsefullNeoException;
import br.com.linkcom.neo.util.ReflectionCache;
import br.com.linkcom.neo.util.ReflectionCacheFactory;
import br.com.linkcom.neo.util.Util;

/**
 * Auxiliador de construo de queries do hibernate
 * 
 * @author rogelgarcia
 *
 * @param <E>
 */
public class QueryBuilder<E> {
    private static final Log log = LogFactory.getLog(QueryBuilder.class);

	private Select select;
	private From from;
	private List<Join> joins = new ArrayList<Join>();
    private List<String> fetchs = new ArrayList<String>();
	private Where where = new Where();
	private GroupBy groupBy;
	private String orderby;
	
	private String alias;
	private HibernateTemplate hibernateTemplate;
	
	protected int maxResults = Integer.MIN_VALUE;
	protected int firstResult = Integer.MIN_VALUE;
	
	protected boolean hasSelect = false;
	
	/**
	 * Utilizar o tradutor quando o resultado for do tipo array
	 */
	protected boolean useTranslator = true;

	protected String translatorAlias;
		
	
	public boolean isUseTranslator() {
		return useTranslator;
	}


	public QueryBuilder<E> setUseTranslator(boolean useTranslator) {
		this.useTranslator = useTranslator;
		return this;
	}
	
	public QueryBuilder<E> setTranslatorAlias(String alias) {
		this.translatorAlias = alias;
		return this;
	}


	public ListagemResult<E> pageAndOrder(FiltroListagem filtroListagem) {
		return new ListagemResult<E>(this, filtroListagem);
	}
	
	
	/**
	 * Configura o Tamanho da pgina e o nmero da pgina atual
	 * Zero Based Index
	 * @param pageNumber numero da pgina que quiser (zero based index)
	 * @param pageSize tamanho de cada pgina (nmero mximo de registros que sero retornados)
	 */
	public QueryBuilder<E> setPageNumberAndSize(int pageNumber, int pageSize){
		maxResults = pageSize;
		firstResult = pageSize * pageNumber;
		return this;
	}
	
	/**
	 * Cria um query builder com o hibernateTemplate
	 * @param hibernateTemplate
	 */
	public QueryBuilder(HibernateTemplate hibernateTemplate){
		if (hibernateTemplate == null) {
            throw new NullPointerException("O construtor do QueryBuilder foi chamado com hibernateTemplate igual a null!");
        }
        this.hibernateTemplate = hibernateTemplate;
	}

	/**
	 * Seta a clusula from desse queryBuilder
	 * Seta o alias para o nome da clase minusculo
	 * @param clazz
	 * @return
	 */
	public QueryBuilder<E> from(Class clazz){
		return from(clazz, StringUtils.uncapitalize(clazz.getSimpleName()));
	}
	
	/**
	 * Seta a clusula from desse queryBuilder e o alias
	 * O alias pode ser utilizado em clausulas where por exemplo
	 * 
	 * @param _clazz
	 * @param _alias
	 * @return
	 */
	public QueryBuilder<E> from(Class _clazz, String _alias){
		if (!hasSelect) {
			select(_alias);
		}
		this.from = new From(_clazz);
		this.alias = _alias;
		return this;
	}
	
	/**
	 * Cria uma clusula from a partir de outra clusula from
	 * @param from
	 * @return
	 */
	public QueryBuilder<E> from(From from){
		from(from.getFromClass(), StringUtils.uncapitalize(from.getFromClass().getSimpleName()));
		return this;
	}

	/**
	 * Cria uma clausula select (opicional)
	 * @param select
	 * @return
	 */
	public QueryBuilder<E> select(String select){
		this.select = new Select(select);
		hasSelect = true;
		return this;
	}
	/**
	 * Cria uma clausula select (opicional)
	 * @param select
	 * @return
	 */
	public QueryBuilder<E> select(Select select){
		this.select = select;
		hasSelect = true;
		return this;
	}

	/**
	 * Inicializa determinada coleo dos beans que essa query retornar
	 * IMPORTANTE: No utilizar o alias, apenas o nome da propriedade que deve ser um Collection
	 * e deve estar mapeado no hibernate
	 * @param path
	 * @return
	 */
	public QueryBuilder<E> fetchCollection(String path){
		fetchs.add(path);
		return this;
	}
	
	/**
	 * Cria uma clausula join
	 * @param joinMode Tipo do join: INNER, LEFT, RIGHT
	 * @param fetch Indica se a entidade relacionada com o join  para ser inicializada
	 * @param path PAth da propriedade, utilizar o alias + ponto + propriedade. ex.: pessoa.municipio
	 * @return
	 */
	public QueryBuilder<E> join(JoinMode joinMode, boolean fetch, String path){
		Join join = new Join(joinMode, fetch, path);
		this.joins.add(join);
		return this;
	}
	/**
	 * Efetua um inner join sem fetch 
	 * Equivalente : join(JoinMode.INNER, false, path);
	 * @param path
	 * @return
	 */
    public QueryBuilder<E> join(String path){
        return join(JoinMode.INNER, false, path);
    }

    /**
     * Efetua um inner join com fetch
     * Equivalente : join(JoinMode.INNER, true, path);
     * @param path
     * @return
     */
    public QueryBuilder<E> joinFetch(String path){
        return join(JoinMode.INNER, true, path);
    }

    /**
     * Efetua um left outer join sem fetch
     * Equivalente : join(JoinMode.LEFT_OUTER, false, path);
     * @param path
     * @return
     */
    public QueryBuilder<E> leftOuterJoin(String path){
        return join(JoinMode.LEFT_OUTER, false, path);
    }
	
    /**
     * Efetua um left outer join com fetch
     * Equivalente : join(JoinMode.LEFT_OUTER, false, path);
     * @param path
     * @return
     */
	public QueryBuilder<E> leftOuterJoinFetch(String path){
		return join(JoinMode.LEFT_OUTER, true, path);
	}

	private boolean inOr = false;
	private boolean bypassedlastWhere = false;
	
	/**
	 * Cria uma clusulua where ... like ...
	 * S  necessrio informar a expressao que deve ser usado o like, no utilizar '?'
	 * Ex.:
	 * whereLike("associado.nome", associado.getNome())
	 * Isso ser transformado em: associado.nome like '%'||nome||'%'
	 * Se o parametro for null ou string vazia essa condio no ser criada
	 * @param whereClause
	 * @param parameter
	 * @return
	 */
	public QueryBuilder<E> whereLike(String whereClause, String parameter) {
		if (parameter != null && !parameter.equals("")) {
			if (parameter.indexOf('?') > 0) {
				throw new IllegalArgumentException("A clusula where do QueryBuilder no pode ter o caracter '?'. Deve ser passada apenas a expresso que se dejesa fazer o like. Veja javadoc!");
			}
			where(whereClause + " like ?", "%" + parameter + "%");
		}
		return this;
	}
	

	public QueryBuilder<E> whereIntervalMatches(String beginfield, String endfield, Object begin, Object end) {
		this.openParentheses()
			.openParentheses()
				.where(beginfield+" >= ?", begin)
				.where(endfield+" <= ?", end)
			.closeParentheses()
			.or()
			.openParentheses()
				.where(beginfield+" >= ?", begin)
				.where(beginfield+" <= ?", end)
			.closeParentheses()
			.or()
			.openParentheses()
				.where(endfield+" >= ?", begin)
				.where(endfield+" <= ?", end)
			.closeParentheses()
			.or()
			.openParentheses()
				.where(beginfield+" <= ?", begin)
				.where(endfield+" >= ?", end)
			.closeParentheses()
		.closeParentheses();
		return this;
	}

    /**
     * Cria uma clusulua where ... like ... ignorando caixa e acentos.
     * S  necessrio informar a expressao que deve ser usado o like, no utilizar '?'
     * Ex.:
     * whereLikeIgnoreAll("associado.nome", associado.getNome())
     * Isso ser transformado em: UPPER(TIRAACENTO(associado.nome)) LIKE '%'||UPPER(TIRAACENTO(nome))||'%'
     * Se o parametro for null ou string vazia essa condio no ser criada
     * @param whereClause
     * @param parameter
     * @return
     */
    public QueryBuilder<E> whereLikeIgnoreAll(String whereClause, String parameter) {
		if (parameter != null && !parameter.equals("")) {
			if (parameter.indexOf('?') > 0) {
				throw new IllegalArgumentException("A clusula where do QueryBuilder no pode ter o caracter '?'. Deve ser passada apenas a expresso que se deseja fazer o like. Veja javadoc!");
			}
			String funcaoTiraacento = Neo.getApplicationContext().getConfig().getProperties().getProperty("funcaoTiraacento");
			if (funcaoTiraacento != null) {
				where("UPPER(" + funcaoTiraacento + "(" + whereClause + ")) LIKE '%'||?||'%'", Util.strings.tiraAcento(parameter).toUpperCase());
			}
			else {
				where("UPPER(" + whereClause + ") LIKE '%'||?||'%'", Util.strings.tiraAcento(parameter).toUpperCase());
			}
		}
		return this;
	}

	/**
	 * Cria uma clausula where ... in ... S  necessrio informar a expressao que deve ser usado o in, no utilizar '?' Ex.: whereIn("associado.inscricoes", inscricoes) Isso ser transformado em: associado.inscricoes in ? [onde no lugar de '?' ser colocado a colecao] Se o parametro for null ou
	 * string vazia e emptyCollectionReturnFalse for false essa condio no ser criada
	 * 
	 * @param whereClause
	 * @param collection
	 * @param emptyCollectionReturnTrue
	 *            Se a colecao for vazia  para retornar verdadeiro?
	 * @return
	 */
	public QueryBuilder<E> whereIn(String whereClause, Collection<?> collection, boolean emptyCollectionReturnTrue) {
		if (collection != null) {
			if (collection.isEmpty()) {
				if (!emptyCollectionReturnTrue) {
					where("1 = 0");
				}
			}
			else {
				where(whereClause + " in (?)", collection);
			}
		}
		else if (!emptyCollectionReturnTrue) {
			where("1 = 0");
		}
		return this;
	}
	
	
	/**
	 * Cria uma clausula where ... in ...
	 * S  necessrio informar a expressao que deve ser usado o in, no utilizar '?'
	 * Ex.:
	 * whereIn("associado.inscricoes", "1,2,4,5")
	 * Isso ser transformado em: associado.inscricoes in ? [onde no lugar de '?' ser colocado o values]
	 * Se o parametro for null ou string vazia e emptyCollectionReturnTrue for true essa condio no ser criada
	 * @param whereClause 
	 * @param values valores separados por virgula
	 * @param emptyCollectionReturnTrue Se a colecao for vazia  para retornar verdadeiro?
	 * @return
	 */
	public QueryBuilder<E> whereIn(String whereClause, String values, boolean emptyCollectionReturnTrue) {
		if (values != null) {
			if (StringUtils.isEmpty(values)) {
				if (!emptyCollectionReturnTrue) {
					where("1 = 0");
				}
			}
			else {
				where(whereClause + " in (" + values + ")");
			}
		}
		else if (!emptyCollectionReturnTrue) {
			where("1 = 0");
		}
		return this;
	}
	
	/**
	 * Cria uma clausula where ... in ...
	 * S  necessrio informar a expressao que deve ser usado o in, no utilizar '?'
	 * Ex.:
	 * whereIn("associado.inscricoes", "1,2,4,5")
	 * Isso ser transformado em: associado.inscricoes in ? [onde no lugar de '?' ser colocado o values]
	 * Se o parametro for null ou string vazia essa condio no ser criada
	 * @param whereClause 
	 * @param values valores separados por virgula
	 * @param emptyCollectionReturnTrue Se a colecao for vazia  para retornar verdadeiro?
	 * @return
	 */
	public QueryBuilder<E> whereIn(String whereClause, String values) {
		return whereIn(whereClause, values, true);
	}
	
	/**
	 * Cria uma clausula where ... in ...
	 * S  necessrio informar a expressao que deve ser usado o in, no utilizar '?'
	 * Ex.:
	 * whereIn("associado.inscricoes", inscricoes)
	 * Isso ser transformado em: associado.inscricoes in ? [onde no lugar de '?' ser colocado a colecao]
	 * Se a colecao for vazia a query retornar verdadeiro
	 * @param whereClause 
	 * @param collection
	 * @return
	 */
	public QueryBuilder<E> whereIn(String whereClause, Collection<?> collection) {
		return whereIn(whereClause, collection, true);
	}
	
	
	public QueryBuilder<E> where(String whereClause, Object parameter, boolean addClause) {
		if(addClause){
			where(whereClause, parameter);
		} else {
			bypassedlastWhere = true;
		}
		if(inOr){
			inOr = false;
		}
		return this;
	}
	/**
	 * Cria uma clausula where.
	 * Escrever a clausula completa . Ex:
	 * where("pessoa.municipio.nome like '?'", nome)
	 * Onde a ? ser substituida por parameter
	 * Se parameter for null essa clausula nao ser criada
	 * @param whereClause
	 * @param parameter
	 * @return
	 */
	public QueryBuilder<E> where(String whereClause, Object parameter) {
		if (parameter != null && (!(parameter instanceof String) || !parameter.equals(""))) {
			if (inOr) {
				where.or();
				inOr = false;
			}
			else {
				where.and();
			}
			where.append(whereClause, parameter);
			bypassedlastWhere = false;
		} else {
			bypassedlastWhere = true;
		}
		if(inOr){
			inOr = false;
		}
		return this;
	}
	
	/**
	 * Cria uma clausula where com vrios parmetros.
	 * Se parameter for null essa clausula nao ser criada
	 * @param whereClause
	 * @param parameter
	 * @return
	 */
	public QueryBuilder<E> where(String whereClause, Object[] parameters) {
		if(parameters == null){
			return this;
		}
		boolean allBlank = true;
		for (int i = 0; i < parameters.length; i++) {
			if (parameters[i] != null && (!(parameters[i] instanceof String) || !parameters[i].equals(""))) {
				allBlank = false;
				break;
			}
		}

		if (!allBlank) {
			if (inOr) {
				where.or();
				inOr = false;
			}
			else {
				where.and();
			}
			where.append(whereClause, parameters);
			bypassedlastWhere = false;
		} else {
			bypassedlastWhere = true;
		}
		return this;
	}
	
	/**
	 * Cria uma clausula where.
	 * Escrever a clausula completa . Ex:
	 * where("pessoa.municipio.nome like 'BH'")
	 * @param whereClause
	 * @param parameter
	 * @return
	 */
	public QueryBuilder<E> where(String whereClause) {
		if (inOr) {
			where.or();
			inOr = false;
		}
		else {
			where.and();
		}
		where.append(whereClause);
		bypassedlastWhere = false;
		return this;
	}
	
	public QueryBuilder<E> where(String whereClause, boolean addClause){
		if(addClause){
			where(whereClause);
		}
		return this;
	}
	
    public QueryBuilder<E> whereWhen(String whereClause, Boolean include) {
		if (include != null && include == true) {
			if (inOr) {
				where.or();
				inOr = false;
			}
			else {
				where.and();
			}
			where.append(whereClause);
			bypassedlastWhere = false;
		} else {
			bypassedlastWhere = true;
		}
		return this;
	}

    public QueryBuilder<E> where(Where where) {
		this.where = where;
		return this;
	}
	
	public QueryBuilder<E> orderBy(String order) {
		if (order != null && !order.trim().equals("")) {
			this.orderby = order;
		}
		return this;
	}
	
	private int subConditionStack = 0;
	
	/**
	 * Cria um "abre parenteses" na query
	 * @return
	 */
	public QueryBuilder<E> openParentheses(){
		if(inOr){
			where.or();
			inOr = false;
		} else {
			where.and();	
		}
		where.append("(");
		subConditionStack++;
		return this;
	}
	
	/**
	 * "Fecha parenteses" na query
	 * @return
	 */
	public QueryBuilder<E> closeParentheses(){
		if(inOr){
			inOr = false;
		}
		bypassedlastWhere = false;
		where.append(")");
		subConditionStack--;
		if(subConditionStack<0){
			throw new RuntimeException("No existem subcondicoes a serem fechadas");
		}
		return this;
	}
		
	/**
	 * Cria uma condio or. A prxima instruo ser concatenada a query com um or e nao com um and
	 * @see openParentheses
	 * @return
	 */
	public QueryBuilder<E> or(){
		if(!bypassedlastWhere){
			inOr = true;	
		}
		return this;
	}
	
	/**
	 * Adiciona ao where uma clusula onde o id deve ser igual ao 
	 * Serializable fornecido
	 * @param serializable
	 * @return
	 */
	public QueryBuilder<E> idEq(Serializable serializable){
		String idPropertyName = hibernateTemplate.getSessionFactory().getClassMetadata(from.getFromClass()).getIdentifierPropertyName();
		where(alias+"."+idPropertyName+" = ?",serializable);
		return this;
	}
	
	/**
	 * Descobre o id do objeto e adiciona uma clusula where
	 * que retornar o objeto com o mesmo id do objeto fornecido
	 * @param object
	 * @return
	 */
	public QueryBuilder<E> entity (Object object){
		if(object == null) throw new NullPointerException("entity null");
		Serializable id = hibernateTemplate.getSessionFactory().getClassMetadata(from.getFromClass()).getIdentifier(object, EntityMode.POJO);
		idEq(id);
		return this;
	}
	
	public QueryBuilder<E> groupBy (String groupBy, String having){
		this.groupBy = new GroupBy(groupBy, having);
		return this;
	}
	
	public QueryBuilder<E> groupBy (String groupBy){
		this.groupBy = new GroupBy(groupBy, null);
		return this;
	}
	
	@SuppressWarnings("unchecked")
	/**
	 * Executa a query e retorna a lista
	 */
	public List<E> list(){
		QueryBuilderResultTranslator qbt = getQueryBuilderResultTranslator();
		Object execute = hibernateTemplate.execute(new HibernateCallback(){
			public Object doInHibernate(Session session) throws HibernateException, SQLException {
				Query query = createQuery(session);
				if(maxResults != Integer.MIN_VALUE){
					query.setMaxResults(maxResults);
				}
				if(firstResult != Integer.MIN_VALUE){
					query.setFirstResult(firstResult);
				}
				List<E> list = query.list();
				
				try {
					for (Object object : list) {
						initializeProxys(object);
					}
				} catch (RuntimeException e) {
					StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
					throw new RuntimeException("Erro ao inicializar Proxys (Colees). "+stackTrace[7],e);
				}
				
				LinkedHashSet<E> linkedHashSet = new LinkedHashSet<E>(list);
				ArrayList<E> arrayList = new ArrayList<E>(linkedHashSet); 
				
				return arrayList;
			}
		}
		);
		if (qbt != null) {
			execute = organizeListWithResultTranslator(qbt, (List)execute);
		}
		return (List<E>) execute;
	}
	
	@SuppressWarnings("unchecked")
	/**
	 * Excecuta a query e retorna um iterator
	 */
	public Iterator<E> iterate(){
		final Session session = SessionFactoryUtils.getNewSession(hibernateTemplate.getSessionFactory(), hibernateTemplate.getEntityInterceptor());
		
		Query query = createQuery(session);
		if(maxResults != Integer.MIN_VALUE){
			query.setMaxResults(maxResults);
		}
		if(firstResult != Integer.MIN_VALUE){
			query.setFirstResult(firstResult);
		}
		final Iterator<E> iterator = query.iterate();
		Iterator<E> closeSessionIterator = new Iterator<E>(){

			public boolean hasNext() {
				boolean hasNext = iterator.hasNext();
				if(!hasNext){
					if (session.isConnected()) {
						session.close();
						//System.out.println("closing connection");
					}
				}
				return hasNext;
			}

			public E next() {
				E next = iterator.next();
				initializeProxys(next);
				return next;
			}

			public void remove() {
				iterator.remove();
			}

			@Override
			protected void finalize() throws Throwable {
				if (session.isConnected()) {
					session.close();
					//System.out.println("closing finalize");
				}
			}
			
		};
		return closeSessionIterator;
	}	
	
	@SuppressWarnings("unchecked")
	/**
	 * Exceuta a query e retorna um nico resultado
	 */
	public E unique(){
		QueryBuilderResultTranslator qbt = getQueryBuilderResultTranslator();
		final boolean useUnique = qbt == null;
		Object execute = hibernateTemplate.execute(new HibernateCallback(){
					public Object doInHibernate(Session session) throws HibernateException, SQLException {
						Query query = createQuery(session);
						if(maxResults != Integer.MIN_VALUE){
							if(maxResults != 1){
								throw new IllegalArgumentException("Para usar o mtodo unique do "+QueryBuilder.class.getSimpleName()+" o maxResults deve ser 1, ou ento no deve ser modificado");
							}
							query.setMaxResults(maxResults);
						}
						if(firstResult != Integer.MIN_VALUE){
							query.setFirstResult(firstResult);
						}
						Object uniqueResult;
						if(useUnique){
							uniqueResult = query.uniqueResult();	
						} else {
							uniqueResult = query.list();
						}
						
						try {
							initializeProxys(uniqueResult);
						} catch (RuntimeException e) {
							StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
							throw new RuntimeException("Erro ao inicializar Proxys (Colees). "+stackTrace[7],e);
						}
						session.evict(uniqueResult);
						return uniqueResult;
					}
				}
				);
		if (qbt != null) {//ORGANIZAR.. TEM 2 LUGARES COM CDIGO IGUAL
			if(execute instanceof List){
				execute = organizeListWithResultTranslator(qbt, (List)execute);
				if(((List)execute).size() == 0){
					execute = null;
				} else {
					execute = ((List)execute).get(0);	
				}
			} else {
				execute = organizeUniqueResultWithTranslator(qbt, (Object[])execute);				
			}
		}
		return (E) execute;
	}


	private Object organizeUniqueResultWithTranslator(QueryBuilderResultTranslator qbt, Object[] execute) {
		Object[] resultado = execute;
		Object novoResultado = qbt.translate((Object[]) resultado);
		return novoResultado;
	}


	@SuppressWarnings("unchecked")
	private Object organizeListWithResultTranslator(QueryBuilderResultTranslator qbt, List execute) {
		List resultado = execute;
		List novoResultado = new ArrayList(resultado.size());
		for (int j = 0; j < resultado.size(); j++) {
			Object translate = qbt.translate((Object[]) resultado.get(j));
			if(translate != null){
				novoResultado.add(translate);
			}
		}
		execute = novoResultado;
		return execute;
	}

	/**
	 * Joins que sero ignorados pelo translator
	 */
	private Set<String> ignoreJoinPaths = new HashSet<String>();
	
	/**
	 * Faz com que o resultTranslator ignore joins com determinados alias
	 * @param alias
	 */
	public void ignoreJoin(String alias){
		ignoreJoinPaths.add(alias);
	}
	
	private QueryBuilderResultTranslator getQueryBuilderResultTranslator() {
		QueryBuilderResultTranslator qbt = null;
		if(useTranslator){//ORGANIZAR TEM 2 LUGARES COM CDIGO IGUAL
			String selectString = select.select;
			if(selectString.contains(".")){
				//utilizar o query builder translator
				qbt = new QueryBuilderResultTranslatorImpl();
				if(translatorAlias != null){
					qbt.setResultAlias(translatorAlias);
				}
				List<AliasMap> aliasMaps = new ArrayList<AliasMap>();
				aliasMaps.add(new AliasMap(this.alias, null, from.getFromClass()));
				for (Join join : joins) {
					String[] joinpath = join.path.split(" +");
					if(join.fetch){
						throw new UsefullNeoException(" necessrio utilizar joins sem Fetch quando especificar os campos a serem selecionados. Erro no join: "+join);
					}
					if(joinpath.length < 2){
						throw new UsefullNeoException(" necessrio informar um alias para todos os joins quando especificar os campos a serem selecionados. Erro no join: "+join);
					}
					if(ignoreJoinPaths.contains(joinpath[1])){
						//se o alias deve ser ignorado.. nao adicionar o aliasmap
						continue;
					}
					aliasMaps.add(new AliasMap(joinpath[1], joinpath[0], null));
				}
				int indexOfDistinct = selectString.indexOf("distinct ");
				if(indexOfDistinct >= 0){
					if(selectString.substring(0, indexOfDistinct).trim().equals("")){//verificando se distinct  a primeira palavra
						selectString = selectString.substring(indexOfDistinct+9);	
					}
				}
				String[] properties = selectString.split("( )*?,( )*?");
				for (int j = 0; j < properties.length; j++) {
					String property = properties[j];
					if(!property.trim().matches("[^ ]+\\.[^ ]+( +as +[^ ]+)?")){
						throw new RuntimeException("O campo \"" + property + "\" do select no  vlido.");
					}
					int indexOfAs = property.indexOf(" as");
					if(indexOfAs > 0){
						properties[j] = property.substring(0, indexOfAs);
					}   
					properties[j] = properties[j].trim();
				}
				qbt.init(properties, aliasMaps.toArray(new AliasMap[aliasMaps.size()]));
				//os extra fields so campos que o QueryBuilderResultTranslator necessita
				String[] extraFields = qbt.getExtraFields();
				String extraFieldsSelect = "";
				for (int j = 0; j < extraFields.length; j++) {
					String extra = extraFields[j];
					extraFieldsSelect += ", ";
					extraFieldsSelect += extra;
				}
				select.select += extraFieldsSelect;
			}
		}
		return qbt;
	}

	private void initializeProxys(Object object) {
		ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
		for (String fetch : fetchs) {
			try {
				Method method = reflectionCache.getMethod(object.getClass(), "get" + StringUtils.capitalize(fetch));
				Object proxy = method.invoke(object);
				Hibernate.initialize(proxy);
			} catch (NoSuchMethodException e){
				throw new UsefullNeoException("Erro ao tentar fazer fetch de "+fetch+" em "+from.getFromClass().getName()+". \nA classe "+from.getFromClass().getName()+" no possui getter para a propriedade '"+fetch+"'", e);
			} catch (Exception e) {
				throw new UsefullNeoException("Erro ao tentar fazer fetch de "+fetch+" em "+from.getFromClass().getName(), e);
			}
		}
	}

	private Query createQuery(Session session) {
		Query query;
		String queryString = null;
		try {
			queryString = getQuery();
			query = session.createQuery(queryString);
		} catch (NullPointerException e) {
			throw new RuntimeException("NullPointerException ao criar query \n\t"+queryString, e);
		}
		for (int i = 0; i < where.getParameters().size(); i++) {
			query.setParameter(i, where.getParameters().get(i));
		}
		return query;
	}
	
	/**
	 * Cria uma query do hibernate com os parametros passados
	 * @return
	 */
	public String getQuery(){
		//TODO VERIFICAR SE A QUERY FOI CONSTRUIDA DE FORMA CORRETA
		if(from == null){
			throw new RuntimeException("No foi informada a clusula from da Query");
		}
		StringBuilder stringBuilder = new StringBuilder();
		stringBuilder.append(select);
		stringBuilder.append(" ");
		stringBuilder.append(from);
		stringBuilder.append(" ");
		for (Join join : joins) {
			stringBuilder.append(join);
			stringBuilder.append(" ");
		}
		
		stringBuilder.append(where);
		
		if(groupBy != null){
			stringBuilder.append(" ");
			stringBuilder.append(groupBy);
		}
		
		if(orderby!=null){
			stringBuilder.append(" ORDER BY ");
			stringBuilder.append(orderby);
		}
		String hibernateQuery = stringBuilder.toString();
		
        log.info(hibernateQuery);
        return hibernateQuery;
	}

	public HibernateTemplate getHibernateTemplate() {
		return hibernateTemplate;
	}

	public class Select{
		private String select;
		public Select(String select) {	this.select = select;}
		public String toString(){return "SELECT "+select;}
		public String getValue() {return select;}
	}
    
	public class From{
		private Class from;
		public Class getFromClass() {return from;}
		public From(Class from){this.from = from;}
		public String toString(){return "FROM "+from.getName()+" "+alias;}
	}
	
	public class Join{
		private JoinMode joinMode;
		private boolean fetch;
		private String path;
		public Join(JoinMode mode, boolean fetch, String path) {
			joinMode = mode;
			this.fetch = fetch;	
			this.path = path;
		}
		public boolean isFetch() {return fetch;	}
		public JoinMode getJoinMode() {	return joinMode;}
		public String getPath() {return path;}
		public String toString(){
            String stringJoinMode;
            if (joinMode.equals(JoinMode.LEFT_OUTER)) {
                stringJoinMode = "LEFT OUTER";
            } else if (joinMode.equals(JoinMode.RIGHT_OUTER)) {
                stringJoinMode = "RIGHT OUTER";
            } else if (joinMode.equals(JoinMode.INNER)) {
                stringJoinMode = "INNER";
            }
            else {
                throw new RuntimeException("Constante JoinMode." + joinMode + " desconhecida.");
            }
            
			return " "+ stringJoinMode+" JOIN "+(fetch?"FETCH ":"")+path;
		}
		public String dontFetchToString(){
			return " "+joinMode+" JOIN "+path;
		}
	}
	
	public class GroupBy {
		private String groupBy;
		private String having;
		
		public GroupBy(String groupBy, String having){
			this.groupBy = groupBy;
			this.having = having;
		}
		
		public String toString(){
			String string = "GROUP BY "+groupBy;
			if(having != null){
				string += " HAVING "+having;
			}
			return string;
		}
	}
	
	public class Where{		
		private StringBuilder stringBuilder = new StringBuilder();
		private List<Object> parameters = new ArrayList<Object>();
		public List<Object> getParameters() {
			return parameters;
		}
		public void or() {if(stringBuilder.length()>0 && !inParentesis)append(" OR "); }
		public void and(){if(stringBuilder.length()>0 && !inParentesis)append(" AND ");}
		public void append(String string, Object parameter){inParentesis = false;stringBuilder.append(string); parameters.add(parameter);}
		public void append(String string, Object[] parameters){inParentesis = false;stringBuilder.append(string); for (Object parameter : parameters) {this.parameters.add(parameter);}}
		boolean inParentesis;
		public void append(String string){
			if(")".equals(string) && inParentesis){
				inParentesis = stringBuilder.substring(stringBuilder.length()-1, stringBuilder.length()).equals("( ");
				
				//se abriu e fechou parenteses .. cancelar os parenteses
				stringBuilder.delete(stringBuilder.length() -2 , stringBuilder.length());
				stringBuilder.append(" 1=1 ");
				
				
				return;
			}
			if("(".equals(string)){
				inParentesis = true;
			} else {
				inParentesis = false;
			}
			stringBuilder.append(string);stringBuilder.append(" ");
		}
		public String toString(){
			if(stringBuilder.length() == 0){
				return "";
			} else {
				StringBuilder builder = new StringBuilder();
				builder.append("WHERE ");
				builder.append(stringBuilder);
				return builder.toString();	
			}
		}
		
		@Override
		public QueryBuilder.Where clone() {
			Where clone = new Where();
			clone.stringBuilder.append(this.stringBuilder.toString());
			clone.parameters.addAll(this.parameters);
			return clone;
		}
	}

	public From getFrom() {
		return from;
	}

	public void setFrom(From from) {
		this.from = from;
	}

	public List<Join> getJoins() {
		return joins;
	}

	public void setJoins(List<Join> joins) {
		this.joins = joins;
	}

	public Select getSelect() {
		return select;
	}

	public void setSelect(Select select) {
		this.select = select;
	}

	public Where getWhere() {
		return where;
	}

	public void setWhere(Where where) {
		this.where = where;
	}

	public String getAlias() {
		return alias;
	}


	public QueryBuilder<E> setFirstResult(int firstResult) {
		this.firstResult = firstResult;
		return this;
	}


	public QueryBuilder<E> setMaxResults(int maxResults) {
		this.maxResults = maxResults;
		return this;
	}


}
