/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.persistence;

import java.lang.annotation.Annotation;
import java.lang.ref.WeakReference;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;

import javax.persistence.Entity;

import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.orm.hibernate3.HibernateTemplate;
import org.springframework.orm.hibernate3.SessionFactoryUtils;
import org.springframework.orm.hibernate3.support.HibernateDaoSupport;
import org.springframework.transaction.support.TransactionTemplate;

import br.com.linkcom.neo.bean.BeanDescriptor;
import br.com.linkcom.neo.bean.BeanDescriptorFactoryImpl;
import br.com.linkcom.neo.bean.annotation.DescriptionProperty;
import br.com.linkcom.neo.controller.crud.FiltroListagem;
import br.com.linkcom.neo.core.standard.Neo;
import br.com.linkcom.neo.util.ReflectionCache;
import br.com.linkcom.neo.util.ReflectionCacheFactory;
import br.com.linkcom.neo.util.Util;



/**
 * Classe que deve ser extendida pelas classes que necessitam fazer acesso ao banco de dados
 * @author rogelgarcia
 *
 * @param <BEAN>
 */
public class GenericDAO<BEAN> extends HibernateDaoSupport {
	
	protected Class<BEAN> beanClass;
	protected String orderBy;
	protected TransactionTemplate transactionTemplate;
	private JdbcTemplate jdbcTemplate;
	
	public void setJdbcTemplate(JdbcTemplate jdbcTemplate) {
		this.jdbcTemplate = jdbcTemplate;
	}
	
	public JdbcTemplate getJdbcTemplate() {
		return jdbcTemplate;
	}
	
	/**
	 * Cria um saveOrUpdateStrategy e salva o objeto (No executa)
	 * @param entity
	 * @return
	 */
	protected SaveOrUpdateStrategy save(Object entity){
		return new SaveOrUpdateStrategy(getHibernateTemplate(), entity).saveEntity();
	}
	
	protected final List<BEAN> empty() {
		return new ArrayList<BEAN>();
	}
	
	@Override
	protected HibernateTemplate createHibernateTemplate(SessionFactory sessionFactory) {
		if(getHibernateTemplate() != null){
			return getHibernateTemplate();
		}
		return super.createHibernateTemplate(sessionFactory);
	}
	
	public TransactionTemplate getTransactionTemplate() {
		return transactionTemplate;
	}

	public void setTransactionTemplate(TransactionTemplate transactionTemplate) {
		this.transactionTemplate = transactionTemplate;
	}

	public GenericDAO(Class<BEAN> beanClass){
		this.beanClass = beanClass;
		ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
		DefaultOrderBy orderBy = reflectionCache.getAnnotation(this.getClass(), DefaultOrderBy.class);
		if (orderBy != null) {
			this.orderBy = orderBy.value();
		}
	}
	
	@SuppressWarnings("unchecked")
	public GenericDAO(){
		Class[] classes = Util.generics.getGenericTypes(this.getClass());
		if(classes.length != 1){
			//tentar a outra forma de Generics
			{
				classes = Util.generics.getGenericTypes2(this.getClass());
				if(classes.length == 1 && !classes[0].equals(Object.class)){
					beanClass = classes[0];
					ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
					DefaultOrderBy orderBy = reflectionCache.getAnnotation(this.getClass(), DefaultOrderBy.class);
					if (orderBy != null) {
						this.orderBy = orderBy.value();
					}
					return;
				}
			}
			throw new RuntimeException("A classe "+this.getClass().getName()+" deveria especificar um tipo generico BEAN" );
		}
		beanClass = classes[0];
		ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
		DefaultOrderBy orderBy = reflectionCache.getAnnotation(this.getClass(), DefaultOrderBy.class);
		if (orderBy != null) {
			this.orderBy = orderBy.value();
		}
		init();
	}
	
	protected void init(){
		
	}
	
	/**
	 * Cria um QueryBuilder para esse DAO j com o from configurado
	 * (O From pode ser alterado)
	 * @return
	 */
	protected QueryBuilder<BEAN> query() {
		return newQueryBuilder(beanClass)
					.from(beanClass);
	}
	
	/**
	 * Cria um query builder para ser utilizado pelo DAO
	 * @param <E>
	 * @param clazz
	 * @return
	 */
	public <E> QueryBuilder<E> newQueryBuilder(Class<E> clazz){
		return new QueryBuilder<E>(getHibernateTemplate());
	}
	
	/**
	 * Cria um query builder para ser utilizado pelo DAO
	 * @param <E>
	 * @param clazz
	 * @return
	 */
	public <E> QueryBuilder<E> newQueryBuilderWithFrom(Class<E> clazz){
		return new QueryBuilder<E>(getHibernateTemplate())
				.from(beanClass);
	}
	
	protected QueryBuilder<BEAN> queryWithOrderBy() {
		QueryBuilder<BEAN> from = newQueryBuilder(beanClass)
					.from(beanClass);
		if(orderBy != null){
			from.orderBy(orderBy);
		}
		return from;
	}

	public void saveOrUpdate(BEAN bean) {
		SaveOrUpdateStrategy save = save(bean);
		updateSaveOrUpdate(save);
		save.execute();
		getHibernateTemplate().flush();
	}


	public BEAN load(BEAN bean){
		if(bean == null){
			return null;
		}
		return query()
			.entity(bean)
			.unique();
	}
	
	public BEAN loadForEntrada(BEAN bean){
		if(bean == null){
			return null;
		}
		QueryBuilder<BEAN> query = newQueryBuilder(beanClass)
			.from(beanClass)
			.entity(bean);
		updateEntradaQuery(query);
		return query
			.unique();
	}
	




	// campos uteis para fazer cache dos findAlls dos combos
	private long cacheTime = TEMPO_5_SEGUNDOS;
	private static final long TEMPO_5_SEGUNDOS = 5*1000;	
	private QueryBuilderResultTranslator translatorQueryFindForCombo;
	private String queryFindForCombo;
	private long lastRead;
	private WeakReference<List> findForComboCache = new WeakReference<List>(null);
	
	
	@SuppressWarnings("unchecked")
	public List<BEAN> findForCombo(String... extraFields){
		if(extraFields != null && extraFields.length > 0){
			return newQueryBuilder(beanClass)
						.select(getComboSelect(extraFields))
						.from(beanClass)
						.orderBy(orderBy)
						.list();
		} else {
			if(queryFindForCombo == null){
				initQueryFindForCombo();
			}
			List listCached = findForComboCache.get();
			if (listCached == null || (System.currentTimeMillis() - lastRead > cacheTime)) {
				listCached = getHibernateTemplate().find(queryFindForCombo);
				listCached = translatorQueryFindForCombo.translate(listCached);
				findForComboCache = new WeakReference<List>(listCached);
				lastRead = System.currentTimeMillis();	
			}
			return listCached;	
		}
	}
	
	@Deprecated
	public List<BEAN> findForCombo(){
		return findForCombo((String[])null);
	}


	
	//cache
	private Map<Class, String> mapaQueryFindByForCombo = new WeakHashMap<Class, String>();
	private Map<Class, String> mapaQueryFindBy = new WeakHashMap<Class, String>();
	
	/**
	 * @param o
	 * @param forCombo indica se  para carregar somente o id e o descriptionProperty
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public List<BEAN> findBy(Object o, boolean forCombo, String... extraFields){
		if(o == null){
			return new ArrayList<BEAN>();
		}
		Class<? extends Object> propertyClass = o.getClass();
		while(propertyClass.getName().contains("$$")){
			propertyClass = propertyClass.getSuperclass();
		}
		String queryString = null;
		if((extraFields != null && extraFields.length >0) || 
				(forCombo && (queryString = mapaQueryFindByForCombo.get(propertyClass)) == null) || 
				(!forCombo && (queryString = mapaQueryFindBy.get(propertyClass)) == null)){
			//inicializa a query para essa classe
			//System.out.println("\n\n\nLOADING CLASSE "+this.beanClass+"  PROPRIEDADE CLASSE: "+o.getClass());
			String[] propertiesForClass = findPropertiesForClass(propertyClass);
			if(propertiesForClass.length == 1){// achou uma propriedade
				String alias = Util.strings.uncaptalize(this.beanClass.getSimpleName());
				String property = propertiesForClass[0];
				QueryBuilder qb = queryWithOrderBy();
				qb.where(alias+"."+property+" = ? ", o);
				updateFindByQuery(qb);
				if(forCombo){
					if(extraFields != null && extraFields.length >0){
						//verifcar se precisa fazer joins extras
						int i = 0;
						for (int j = 0; j < extraFields.length; j++) {
							String extra = extraFields[j];
							BeanDescriptor<? extends Object> beanDescriptor = Neo.getApplicationContext().getBeanDescriptor(null, this.beanClass);
							Type type = beanDescriptor.getPropertyDescriptor(extra).getType();
							if(type instanceof Class){
								if(((Class)type).isAnnotationPresent(Entity.class)){
									extra+= "." + Neo.getApplicationContext().getBeanDescriptor(null, (Class)type).getDescriptionPropertyName();
								}
							}
							if(extra.contains(".")){
								int ultimoponto = extra.lastIndexOf(".");
								String path = extra.substring(0, ultimoponto);
								qb.join(alias+"."+path+" autojoin"+i);
								extraFields[j] = "autojoin"+i+extra.substring(ultimoponto);
							}
							i++;
						}
						//se for com extraFields no pode usar o cache (no existe cache do select quando tem extra properties)						
						qb.select(getComboSelect(extraFields));
						String hbquery = qb.getQuery();	
						queryString = hbquery;
						return qb.list();
					} else {
						qb.select(getComboSelect());
						String hbquery = qb.getQuery();	
						queryString = hbquery;
						mapaQueryFindByForCombo.put(propertyClass, queryString);
					}
				} else {
					String hbquery = qb.getQuery();	
					queryString = hbquery;
					mapaQueryFindByForCombo.put(propertyClass, queryString);
				}
			} else if(propertiesForClass.length > 1){// mais de uma propriedade do mesmo tipo
				throw new RuntimeException("No foi possvel executar findBy(..). Existe mais de uma propriedade da classe "+propertyClass.getName()+" na classe "+this.beanClass.getName()+"  "+Arrays.deepToString(propertiesForClass));
			} else {//nenhuma propriedade do tipo fornecido
				throw new RuntimeException("No foi possvel executar findBy(..). No existe nenhuma propriedade da classe "+propertyClass.getName()+" na classe "+this.beanClass.getName());
			}
		}

		List list = getHibernateTemplate().find(queryString, o);
		if (forCombo) {
			initQueryFindForCombo();
			list = translatorQueryFindForCombo.translate(list);
		}

		return list;
	}
	
	/**
	 * Atualiza as querys de findBy
	 * ex.: findBy(Object o, boolean forCombo, String... extraFields)
	 * @param query
	 */
	protected void updateFindByQuery(QueryBuilder query) {
		
		
	}

	private String[] findPropertiesForClass(Class<? extends Object> propertyClass) {
		List<String> properties = new ArrayList<String>();
		ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
		Method[] methods = reflectionCache.getMethods(this.beanClass);
		for (Method method : methods) {
			if(Util.beans.isGetter(method)){
				if(method.getReturnType().equals(propertyClass)){
					properties.add(Util.beans.getPropertyFromGetter(method.getName()));
				}
			}
		}
		return properties.toArray(new String[properties.size()]);
	}



	//cache
	private String comboSelect;
	/**
	 * Retorna o select necessrio para apenas o @Id e o @DescriptionProperty
	 * ex.: new QueryBuilder(...).select(getComboSelect()).from(X.class);
	 * @param extraFields 
	 * @return
	 */
	public String getComboSelect(String... extraFields){
		if(comboSelect == null || extraFields != null && extraFields.length > 0){
			String alias = Util.strings.uncaptalize(beanClass.getSimpleName());
			String[] selectedProperties = getComboSelectedProperties(alias);
			selectedProperties = organizeExtraFields(alias, selectedProperties, extraFields);
			String comboSelect = "";
			for (int i = 0; i < selectedProperties.length; i++) {
				String prop = selectedProperties[i];
				comboSelect = comboSelect + prop;
				if(i+1 < selectedProperties.length){
					comboSelect = comboSelect + ", ";
				}
			}
			if(!(extraFields != null && extraFields.length > 0)){
				this.comboSelect = comboSelect;	
			} else {
				return comboSelect;
			}
		}
		return comboSelect;
	}
	

	protected String[] getComboSelectedProperties(String alias) {
		BeanDescriptor<BEAN> beanDescriptor = new BeanDescriptorFactoryImpl().createBeanDescriptor(null, beanClass);
		String descriptionPropertyName = beanDescriptor.getDescriptionPropertyName();
		String idPropertyName = beanDescriptor.getIdPropertyName();
		String[] selectedProperties;
		if(descriptionPropertyName == null){
			selectedProperties = new String[]{alias+"."+idPropertyName};
		} else {
			//verificar se o descriptionproperty utiliza outros campos
			Annotation[] annotations = beanDescriptor.getPropertyDescriptor(descriptionPropertyName).getAnnotations();
			DescriptionProperty descriptionProperty = null;
			for (Annotation annotation : annotations) {
				if(DescriptionProperty.class.isAssignableFrom(annotation.annotationType())){
					descriptionProperty = (DescriptionProperty) annotation;
					break;
				}
			}
			String[] usingFields = null;
			if(descriptionProperty != null){
				//TODO PROCURAR O @DescriptionProperty nas classes superiores
				usingFields = descriptionProperty.usingFields();
			}
			
			if(usingFields != null && usingFields.length > 0 ){
				selectedProperties = new String[usingFields.length+1];
				selectedProperties[0] = alias+"."+idPropertyName;
				for (int i = 0; i < usingFields.length; i++) {
					selectedProperties[i+1] = alias+"."+usingFields[i];
				}
			} else {
				selectedProperties = new String[]{alias+"."+idPropertyName, alias+"."+descriptionPropertyName};	
			}
			
		}
		return selectedProperties;
	}
	
	private void initQueryFindForCombo() {
		if (translatorQueryFindForCombo == null) {
			//BeanDescriptor<BEAN> beanDescriptor = Neo.getApplicationContext().getBeanDescriptor(null, beanClass);
			//String descriptionPropertyName = beanDescriptor.getDescriptionPropertyName();
			//String idPropertyName = beanDescriptor.getIdPropertyName();

			String alias = Util.strings.uncaptalize(beanClass.getSimpleName());
			String[] selectedProperties = getComboSelectedProperties(alias);
			String hbQueryFindForCombo = "select " + getComboSelect() + " " + "from " + beanClass.getName() + " " + alias;
			hbQueryFindForCombo = getQueryFindForCombo(hbQueryFindForCombo);
			if (orderBy != null) {
				hbQueryFindForCombo += "  order by " + orderBy;
			}
			translatorQueryFindForCombo = new QueryBuilderResultTranslatorImpl();
			translatorQueryFindForCombo.init(selectedProperties, new AliasMap[] { new AliasMap(alias, null, beanClass) });
			queryFindForCombo = hbQueryFindForCombo;
		}
	}

	private String[] organizeExtraFields(String alias, String[] selectedProperties, String... extraFields) {
		if(extraFields != null && extraFields.length > 0){
			for (int i = 0; i < extraFields.length; i++) {
				if (!extraFields[i].contains(".")) {
					extraFields[i] = alias + "." + extraFields[i];
				}
			}
			String[] oldselectedProperties = selectedProperties;
			selectedProperties = new String[selectedProperties.length+extraFields.length];
			System.arraycopy(oldselectedProperties, 0, selectedProperties, 0, oldselectedProperties.length);
			System.arraycopy(extraFields, 0, selectedProperties, oldselectedProperties.length, extraFields.length);
		}
		return selectedProperties;
	}
	
	//TODO RENOMEAR PARA UPDATEFINDFORCOMBOQUERY
	/**
	 * Permite alterar a query que ser utilizada no findForCombo. O parametro  a query montada automaticamente.
	 * Ser feito cache dessa query entao esse mtodo s ser chamado uma vez.
	 * @param hbQueryFindForCombo
	 * @return
	 */
	protected String getQueryFindForCombo(String hbQueryFindForCombo) {
		return hbQueryFindForCombo;
	}

	/**
	 * Retorna uma lista com todos os beans encontrados no banco
	 * @return
	 */
	public List<BEAN> findAll(){
		return findAll(null);
	}
	
	/**
	 * Retorna uma lista com todos os beans encontrados no banco 
	 * ordenados por determinada propriedade
	 * @param orderBy propriedade que deve ser utilizada na ordenao
	 * @return
	 */
	public List<BEAN> findAll(String orderBy){
		return query()
				.orderBy(orderBy)
				.list();
	}
	
	public ListagemResult<BEAN> findForListagem(FiltroListagem filtro){
		QueryBuilder<BEAN> query = query();
		query.orderBy(Util.strings.uncaptalize(beanClass.getSimpleName())+".id");
		updateListagemQuery(query, filtro);
		QueryBuilder<BEAN> queryBuilder = query;
		return new ListagemResult<BEAN>(queryBuilder, filtro);
	}
	
	public void updateListagemQuery(QueryBuilder<BEAN> query, FiltroListagem _filtro) {
	}

	public void updateEntradaQuery(QueryBuilder<BEAN> query) {
	}
	
	public void updateSaveOrUpdate(SaveOrUpdateStrategy save) {
	}

	public void delete(final BEAN bean){
		// verifica se j existe uma transao acontecendo.. se existir.. utiliza a atual
		// se nao .. cria uma nova
		Session session = SessionFactoryUtils.getSession(getHibernateTemplate().getSessionFactory(), getHibernateTemplate().getEntityInterceptor(), getHibernateTemplate().getJdbcExceptionTranslator());
		Transaction transaction = session.getTransaction();
		boolean intransaction = true;
		if(!transaction.isActive()){
			transaction = session.beginTransaction();	
			intransaction = false;
		}
		 
		try {
			
			getHibernateTemplate().delete(bean);
			session.flush();
			if (!intransaction) {
				transaction.commit();
			}
		} catch (HibernateException ex) {
			if (!intransaction) {
				transaction.rollback();
			}
			throw convertHibernateAccessException(ex);
		} catch (RuntimeException e) {
			if (!intransaction) {
				transaction.rollback();
			}
			throw e;
		} catch (Exception e){
			if (!intransaction) {
				transaction.rollback();
			}
			throw new RuntimeException(e);
		} finally {
			
			SessionFactoryUtils.releaseSession(session, getHibernateTemplate().getSessionFactory());
				
		}
		
		
		/*
		// esse cdigo teve que ser feito porque o delete do hibernateTemplate tenta d uns updates quando tem UserType na classe
		// no suportar camposite pk
		// PARECE que foi consertado, se voltar a contecer updates na hora de deltar voltar com o cdigo e arrumar outra estratgia para fazer 
		// o log da operacoes no banco
		final ClassMetadata classMetadata = getHibernateTemplate().getSessionFactory().getClassMetadata(beanClass);
		final String id = classMetadata.getIdentifierPropertyName();
		getHibernateTemplate().execute(new HibernateCallback(){

			public Object doInHibernate(Session session) throws HibernateException, SQLException {
				Transaction transaction = session.beginTransaction();
								
				Serializable identifier = classMetadata.getIdentifier(bean, EntityMode.POJO);
				
				//querys manuais no chamam os listeners.. chamar aqui
								
				session
					.createQuery("delete "+beanClass.getName()+" where "+id+" = :id")
					.setParameter("id", identifier)
					.executeUpdate();
				
				
				transaction.commit();
				return null;
			}
			
		});
		//getHibernateTemplate().delete(bean);
		 */
		 
	}


}
