/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.persistence;

import java.beans.PropertyDescriptor;
import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import br.com.linkcom.neo.bean.annotation.DAOBean;
import br.com.linkcom.neo.core.standard.Neo;
import br.com.linkcom.neo.exception.NeoException;
import br.com.linkcom.neo.types.File;
import br.com.linkcom.neo.util.Util;


@DAOBean
public class FileDAO<BEAN extends File> extends GenericDAO<BEAN> {
	
	public <E extends File> E loadWithContents(E bean) {
		E arquivo = new QueryBuilder<E>(getHibernateTemplate())
							.from(bean.getClass())
							.entity(bean)
							.unique();
		readFile(arquivo);
		return arquivo;
	}

	public void fillWithContents(File file) {
		readFile(file);
	}
	
	protected void readFile(File arquivo) {
		String nomeArquivo = getNomeArquivo(arquivo);
		java.io.File file = new java.io.File(nomeArquivo);
		try {
			InputStream inputStream = new FileInputStream(file);
			BufferedInputStream bufferedInputStream = new BufferedInputStream(inputStream);
			byte[] bytes = new byte[arquivo.getSize().intValue()];
			bufferedInputStream.read(bytes);
			arquivo.setContent(bytes);
		} catch (FileNotFoundException e) {
			throw new NeoException("Arquivo no encontrado. Cdigo: "+arquivo.getCdfile(), e);
		} catch (IOException e) {
			throw new NeoException("No foi possvel ler o arquivo. ", e);
		}
	}
	
	protected String getNomeArquivo(File arquivo) {
		if(arquivo == null) {
			throw new NullPointerException("Arquivo invlido (nulo)");
		}
		if(arquivo.getCdfile() == null) {
			throw new NullPointerException("Id do arquivo invlido");
		}
		String saveDir = getSaveDir();
		return saveDir + java.io.File.separator + "arquivo" + arquivo.getCdfile()+"."+getExtensao();
	}

	protected String getExtensao() {
		return "neo";
	}

	protected String getSaveDir() {
		return System.getProperty("user.home")+java.io.File.separator+"dados"+java.io.File.separator+Neo.getApplicationName()+java.io.File.separator+"arquivos";
	}
	
	public void saveFile(Object bean, String filePropertyName) {
		File arquivoVelho = null;
		File arquivoAtual = null;
		PropertyDescriptor pd;
		try {
			pd = new PropertyDescriptor(filePropertyName, bean.getClass());
			arquivoAtual = (File) pd.getReadMethod().invoke(bean);
		} catch (Exception e) {
			throw new NeoException("No foi possivel adquirir arquivo. Propriedade "+filePropertyName+" da classe "+bean.getClass(), e);
		}
		if(Util.hibernate.getId(getHibernateTemplate(), bean) != null){
			Object beanVelho = new QueryBuilder<Object>(getHibernateTemplate())
						.from(bean.getClass())
						.leftOuterJoinFetch(Util.strings.uncaptalize(bean.getClass().getSimpleName())+"."+filePropertyName+" "+filePropertyName)
						.entity(bean)
						.unique();
			try {
				arquivoVelho = (File) pd.getReadMethod().invoke(beanVelho);
			} catch (Exception e) {
				throw new NeoException("No foi possivel adquirir arquivo. Propriedade "+filePropertyName+" da classe "+bean.getClass(), e);
			}
		}
		
		File save = save(arquivoAtual, arquivoVelho);
		try {
			pd.getWriteMethod().invoke(bean, save);
		} catch (Exception e) {
			throw new NeoException("No foi possvel setar o arquivo. Propriedade "+filePropertyName+" da classe "+bean.getClass(), e);
		}
	}
	
	public File save(File arquivoNovo, File arquivoVelho){
		try {
			if (arquivoVelho == null) {
				// criar
				if (arquivoNovo != null && arquivoNovo.getSize() > 0) {
					getHibernateTemplate().saveOrUpdate(arquivoNovo);
					
					String nomeArquivo = getNomeArquivo(arquivoNovo);
					writeFile(arquivoNovo, nomeArquivo);
				} else {
					return null;
				}
			} else {
				// atualizar
				if(arquivoNovo == null){
					//apagar o arquivo
					getHibernateTemplate().delete(arquivoVelho);
					String nomeArquivo = getNomeArquivo(arquivoVelho);
					deleteFile(nomeArquivo);
				} else if(arquivoNovo.getSize() > 0){ 
					getHibernateTemplate().evict(arquivoVelho);
					arquivoNovo.setCdfile(arquivoVelho.getCdfile());
					//sobrescrever o arquivo
					getHibernateTemplate().saveOrUpdate(arquivoNovo);
					String nomeArquivo = getNomeArquivo(arquivoNovo);
					writeFile(arquivoNovo, nomeArquivo);
				} else {
					//se o tamanho for zero no mexer no arquivo
					arquivoNovo.setCdfile(arquivoVelho.getCdfile());
				}
			}
			getHibernateTemplate().flush();
			return arquivoNovo;
		} catch (Exception e) {
			String name;
			try {
				name = getNomeArquivo(arquivoNovo);
			} catch (Exception e2) {
				name = "(No foi possvel adquirir o nome do arquivo. Erro: "+e2.getMessage()+")";
			}
			throw new NeoException("No foi posvel salvar o arquivo. "+name, e);
		}
	}
	
	protected void deleteFile(String nomeArquivo) {
		java.io.File file = new java.io.File(nomeArquivo);
		file.delete();
	}
	
	protected void writeFile(File arquivoNovo, String nomeArquivo) throws IOException {
		java.io.File file = new java.io.File(nomeArquivo);
		if(!file.exists()){
			file.getParentFile().mkdirs();
			//file.mkdirs();
			file.createNewFile();
		}
		OutputStream out = new FileOutputStream(file);
		out.write(arquivoNovo.getContent());
	}

}
