/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.hibernate;

import java.io.IOException;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.HibernateException;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.AnnotationConfiguration;
import org.hibernate.cfg.Configuration;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.orm.hibernate3.LocalSessionFactoryBean;

import br.com.linkcom.neo.exception.ConfigurationException;

/**
 * @author rogelgarcia
 * @since 04/10/2005
 */
public class AnnotationSessionFactoryBean extends LocalSessionFactoryBean implements ApplicationContextAware {
	
	private static final Log log = LogFactory.getLog(AnnotationSessionFactoryBean.class);
	
	private ApplicationContext applicationContext;
	private String dialect;
	
	protected Class<?>[] annotatedClasses;
	protected EntityFinder entityFinder = new DefaultEntityFinder();
	
	public void setEntityFinder(EntityFinder entityFinder) {
		this.entityFinder = entityFinder;
	}
	
	public EntityFinder getEntityFinder() {
		return entityFinder;
	}
	
	public Class<?>[] getAnnotatedClasses() {
		return annotatedClasses;
	}

	public void setAnnotatedClasses(Class<?>[] annotatedClasses) {
		this.annotatedClasses = annotatedClasses;
	}
	
	public AnnotationSessionFactoryBean(){
		setConfigurationClass(AnnotationConfiguration.class);
	}

	@Override
	protected SessionFactory newSessionFactory(Configuration config) throws HibernateException {
		AnnotationConfiguration annotationConfiguration = (AnnotationConfiguration) config;
		
		configureAnnotatedBeans(annotationConfiguration);
		
		return super.newSessionFactory(config);
	}

	public void configureAnnotatedBeans(AnnotationConfiguration annotationConfiguration) {
		if (entityFinder != null) {
			if(entityFinder instanceof DefaultEntityFinder){
				((DefaultEntityFinder)entityFinder).setApplicationContext(applicationContext);
			}
			Class<?>[] classesWithAnnotationEntity;
			try {
				classesWithAnnotationEntity = entityFinder.findEntities();
			} catch (IOException e) {
				throw new ConfigurationException(e);
			}
			for (Class<?> annotatedClass : classesWithAnnotationEntity) {
				log.debug("Registrando automaticamente ... "	+ annotatedClass.getName());
				annotationConfiguration.addAnnotatedClass(annotatedClass);
			}
		} else {
			log.warn("EntityFinder no setado");
		}
		
		if (annotatedClasses != null){
			for (Class<?> annotatedClass : annotatedClasses) {
				log.debug("Registrando ... "+annotatedClass.getName());
				annotationConfiguration.addAnnotatedClass(annotatedClass);
			}
		}
	}

	public void setApplicationContext(ApplicationContext applicationContext) {
		this.applicationContext = applicationContext;
	}

	/* (non-Javadoc)
	 * @see org.springframework.orm.hibernate3.LocalSessionFactoryBean#setHibernateProperties(java.util.Properties)
	 */
	@Override
	public void setHibernateProperties(Properties hibernateProperties) {
		if(hibernateProperties.getProperty("hibernate.show_sql") == null){
			hibernateProperties.setProperty("hibernate.show_sql", "true");
		}
		setDialect(hibernateProperties.getProperty("hibernate.dialect"));
		super.setHibernateProperties(hibernateProperties);
	}

	/**
	 * @return Returns the dialect.
	 */
	public String getDialect() {
		return dialect;
	}

	/**
	 * @param dialect The dialect to set.
	 */
	public void setDialect(String dialect) {
		this.dialect = dialect;
	}
}
