/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.crypto;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.crypto.KeyGenerator;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;

import br.com.linkcom.neo.exception.NeoException;
import br.com.linkcom.neo.util.Util;

public class MacUtil {

	protected String patternAuthentication = "(.*?)\\[(\\d{8})\\-([A-Za-z0-9\\+/]+={1,2})\\]";
	protected String patternDate = "ddMMyyyy";
	
	/**
	 * Gera uma nova chave de autenticao.
	 * @return
	 * @throws NoSuchAlgorithmException
	 */
	public String generateMacKey() throws NoSuchAlgorithmException {
		KeyGenerator keyGen = KeyGenerator.getInstance("HmacSHA1");
		SecretKey key = keyGen.generateKey();
		byte[] keyArray = key.getEncoded();
		String keyString = new sun.misc.BASE64Encoder().encode(keyArray);
		return keyString;
	}
	
	/**
	 * Autentica um texto e retorna o documento original simplificado e acrescentado da linha de autenticao.
	 * @param message Texto original a ser autenticado.
	 * @param keyString chave para autenticao.
	 * @return
	 * @throws IOException
	 * @throws NoSuchAlgorithmException
	 * @throws InvalidKeyException
	 */
	public String authenticateMessage(String message, String keyString) throws IOException, NoSuchAlgorithmException, InvalidKeyException {
	    String simplifiedMessage = simplifyMessage(message);
	    String oneLineMessage = oneLineMessage(simplifiedMessage);
	    String digestString = generateMac(oneLineMessage, keyString);
		String encapsulatedMessage = encapsulateMessage(simplifiedMessage, digestString);
		return encapsulatedMessage;
	}

	/**
	 * Autentica um texto e retorna a linha de autenticao a ser acrescentado no final do documento original.
	 * @param message Texto original a ser autenticado.
	 * @param keyString chave para autenticao.
	 * @return
	 * @throws IOException
	 * @throws NoSuchAlgorithmException
	 * @throws InvalidKeyException
	 */
	public String generateAuthentication(String message, String keyString) throws IOException, NoSuchAlgorithmException, InvalidKeyException {
	    String simplifiedMessage = simplifyMessage(message);
	    String oneLineMessage = oneLineMessage(simplifiedMessage);
	    String digestString = generateMac(oneLineMessage, keyString);
		String tagLine = generateTagline(digestString);
		return tagLine;
	}

	/**
	 * Valida a autenticao de uma mensagem.
	 * @param message
	 * @param keyString
	 * @throws ParseException 
	 * @throws IOException 
	 * @throws NoSuchAlgorithmException 
	 * @throws InvalidKeyException 
	 */
	public boolean validateMessage(String message, String keyString) throws ParseException, InvalidKeyException, NoSuchAlgorithmException, IOException {
		String msg = getMessage(message);
	    String simplifiedMessage = simplifyMessage(msg);
	    String oneLineMessage = oneLineMessage(simplifiedMessage);
		
		String mac = getMac(message);
	    String generatedMac = generateMac(oneLineMessage, keyString);

	    if (mac.equals(generatedMac)) {
			return true;
		}
	    else {
	    	return false;
	    }
	}
	
	/**
	 * Retorna a mensagem contida na mensagem fornecida.
	 * @param message
	 * @return
	 * @throws ParseException 
	 */
	private String getMessage(String message) throws ParseException {
		Pattern pattern = Pattern.compile(patternAuthentication, Pattern.DOTALL);
		Matcher matcher = pattern.matcher(message);
		if (matcher.find()) {
			String msg = matcher.group(1);
			return msg;
		}
		else {
			throw new NeoException("Mensagem invlida");
		}
	}
	
	/**
	 * Retorna a data contida na mensagem fornecida.
	 * @param message
	 * @return
	 * @throws ParseException 
	 */
	public Date getDate(String message) throws ParseException {
		Pattern pattern = Pattern.compile(patternAuthentication, Pattern.DOTALL);
		Matcher matcher = pattern.matcher(message);
		if (matcher.find()) {
			String dateString = matcher.group(2);
			Date date = new SimpleDateFormat(patternDate).parse(dateString);
			return date;
		}
		else {
			throw new NeoException("Mensagem invlida");
		}
	}
	
	/**
	 * Retorna a mensagem contida na mensagem fornecida.
	 * @param message
	 * @return
	 * @throws ParseException 
	 */
	private String getMac(String message) throws ParseException {
		Pattern pattern = Pattern.compile(patternAuthentication, Pattern.DOTALL);
		Matcher matcher = pattern.matcher(message);
		if (matcher.find()) {
			String mac = matcher.group(3);
			return mac;
		}
		else {
			throw new NeoException("Mensagem invlida");
		}
	}
	
	private String generateMac(String oneLineMessage, String keyString) throws NoSuchAlgorithmException, IOException, InvalidKeyException {
	    byte[] keyArray = new sun.misc.BASE64Decoder().decodeBuffer(keyString);
		SecretKey key = new SecretKeySpec(keyArray, "HmacSHA1");
		Mac mac = Mac.getInstance("HmacSHA1");
	    mac.init(key);
	    byte[] messageArray = oneLineMessage.getBytes("ISO-8859-1");
        byte[] digest = mac.doFinal(messageArray);
        String digestString = new sun.misc.BASE64Encoder().encode(digest);
        
        return digestString;
	}
	
	private String simplifyMessage(String message) {
		message = Util.strings.tiraAcento(message);
		message = message.replaceAll("[^\\p{Graph}\\n\\r]", " ");
		message = message.replaceAll(" +", " ");
		message = message.trim();
		return message;
	}

	private String oneLineMessage(String message) {
	    message = message.replaceAll("[\\n\\r]", " ");
		message = message.replaceAll(" +", " ");
		message = message.trim();
		return message;
	}
	
	private String encapsulateMessage(String simplifiedMessage, String digestString) {
		return simplifiedMessage + "\n" + generateTagline(digestString);
	}
	
	public String generateTagline(String digestString) {
		String date = new SimpleDateFormat(patternDate).format(new Date());
		return "[" + date + "-" + digestString + "]";
	}
	
	public static void main(String[] args) throws NoSuchAlgorithmException, InvalidKeyException, IOException, ParseException {
		String message = "Fabrcio  - 20/08/1978 -> R$ 123.456,78 []{}*()-=_+\nlala\n\nlele";
		System.out.println(message);
		System.out.println("===================");
		
		String keyString = Util.crypto.generateMacKey();
		System.out.println(keyString);
		System.out.println("===================");

		String authenticatedMessage = Util.crypto.authenticateMessage(message, keyString);
		System.out.println(authenticatedMessage);
		System.out.println("===================");
		
		String enteredMessage = "";
		BufferedReader entrada = new BufferedReader(new InputStreamReader(System.in));
		String linha = entrada.readLine();
		while (!linha.equals(".")) {
			enteredMessage += linha + "\n";
			linha = entrada.readLine();
		}
		System.out.println("===================");
		
		String enteredkey = "";
		linha = entrada.readLine();
		while (!linha.equals(".")) {
			enteredkey += linha + "\n";
			linha = entrada.readLine();
		}
		enteredkey = enteredkey.replaceAll("\\s", "");
		System.out.println("===================");

		Date data = Util.crypto.getDate(enteredMessage);
		System.out.println(data);
		System.out.println("===================");

		boolean ok = Util.crypto.validateMessage(enteredMessage, enteredkey);
		System.out.println(ok);
		System.out.println("===================");
	}
	
}
