/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.crypto;

import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.text.ParseException;
import java.util.Date;

import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;

import org.apache.commons.codec.binary.Base64;

public class CryptoUtil {

	MacUtil macUtil = new MacUtil();
	
	/**
	 * Gera uma nova chave de autenticao.
	 * @return
	 * @throws NoSuchAlgorithmException
	 */
	public String generateMacKey() throws NoSuchAlgorithmException {
		return macUtil.generateMacKey();
	}
	
	/**
	 * Autentica um texto e retorna o documento original simplificado e acrescentado da linha de autenticao.
	 * @param message Texto original a ser autenticado.
	 * @param keyString chave para autenticao.
	 * @return
	 * @throws IOException
	 * @throws NoSuchAlgorithmException
	 * @throws InvalidKeyException
	 */
	public String authenticateMessage(String message, String keyString) throws IOException, NoSuchAlgorithmException, InvalidKeyException {
		return macUtil.authenticateMessage(message, keyString);
	}

	/**
	 * Autentica um texto e retorna a linha de autenticao a ser acrescentado no final do documento original.
	 * @param message Texto original a ser autenticado.
	 * @param keyString chave para autenticao.
	 * @return
	 * @throws IOException
	 * @throws NoSuchAlgorithmException
	 * @throws InvalidKeyException
	 */
	public String generateAuthentication(String message, String keyString) throws IOException, NoSuchAlgorithmException, InvalidKeyException {
		return macUtil.generateAuthentication(message, keyString);
	}

	/**
	 * Retorna a data contida na mensagem fornecida.
	 * @param message
	 * @return
	 * @throws ParseException 
	 */
	public Date getDate(String message) throws ParseException {
		return macUtil.getDate(message);
	}
	
	/**
	 * Valida a autenticao de uma mensagem.
	 * @param message
	 * @param keyString
	 * @throws ParseException 
	 * @throws IOException 
	 * @throws NoSuchAlgorithmException 
	 * @throws InvalidKeyException 
	 */
	public boolean validateMessage(String message, String keyString) throws ParseException, InvalidKeyException, NoSuchAlgorithmException, IOException {
		return macUtil.validateMessage(message, keyString);
	}
	
	public static String generateMacBase64(String message, String key) throws NoSuchAlgorithmException, IOException, InvalidKeyException {
	    byte[] messageArray = message.getBytes("ISO-8859-1");
	    byte[] keyArray = key.getBytes("ISO-8859-1");

	    SecretKey secretKey = new SecretKeySpec(keyArray, "HmacSHA1");
		Mac mac = Mac.getInstance("HmacSHA1");
	    mac.init(secretKey);
        byte[] digestArray = mac.doFinal(messageArray);

        byte[] digestBase64Array = Base64.encodeBase64(digestArray);
		String digest = new String(digestBase64Array, "ISO-8859-1");
        return digest;
	}

}
