/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.core.web.init;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.SessionFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.MutablePropertyValues;
import org.springframework.beans.factory.config.AutowireCapableBeanFactory;
import org.springframework.beans.factory.config.BeanFactoryPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.config.RuntimeBeanReference;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.beans.factory.support.RootBeanDefinition;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.support.SQLErrorCodeSQLExceptionTranslator;
import org.springframework.orm.hibernate3.HibernateTemplate;
import org.springframework.orm.hibernate3.HibernateTransactionManager;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.support.TransactionTemplate;

import br.com.linkcom.neo.authorization.AuthorizationDAO;
import br.com.linkcom.neo.classmanager.ClassManager;
import br.com.linkcom.neo.controller.DispatcherServlet;
import br.com.linkcom.neo.controller.NeoCommonsMultipartResolver;
import br.com.linkcom.neo.core.config.Config;
import br.com.linkcom.neo.core.config.DefaultConfig;
import br.com.linkcom.neo.exception.ConfigurationException;
import br.com.linkcom.neo.exception.NeoException;
import br.com.linkcom.neo.hibernate.AnnotationSessionFactoryBean;
import br.com.linkcom.neo.persistence.OracleSQLErrorCodeSQLExceptionTranslator;
import br.com.linkcom.neo.persistence.PostgreSQLErrorCodeSQLExceptionTranslator;
import br.com.linkcom.neo.persistence.SQLServerSQLErrorCodeSQLExceptionTranslator;
import br.com.linkcom.neo.util.NeoFormater;
import br.com.linkcom.neo.util.NeoImageResolver;
import br.com.linkcom.neo.util.Util;

/**
 * @author rogelgarcia
 * @since 22/01/2006
 * @version 1.1
 */
public class NeoBeanFactoryPostProcessor implements BeanFactoryPostProcessor {
	
	protected Map<String, Class> mapaDatabaseSqlErrorTranslators = new HashMap<String, Class>();
	protected Map<String, String> mapaDatabaseDialect = new HashMap<String, String>();
	
	protected static final Log log = LogFactory.getLog(NeoBeanFactoryPostProcessor.class);
	
	private ClassManager classManager;
	/** Indica se o config foi encontrado no XML */
	private boolean configInXML = false;
	private DataSourceConfigStrategy dataSourceConfigStrategy;
	private String configName;
	private boolean authorizationDAOAutoconfig;
	
	public NeoBeanFactoryPostProcessor(ClassManager classManager, DataSourceConfigStrategy dataSourceConfigStrategy){
		this.classManager = classManager;
		this.dataSourceConfigStrategy = dataSourceConfigStrategy;
		registerMaps();
	}

	private void registerMaps() {
		//error codes translators
		mapaDatabaseSqlErrorTranslators.put("PostgreSQL", PostgreSQLErrorCodeSQLExceptionTranslator.class);
		mapaDatabaseSqlErrorTranslators.put("Oracle", OracleSQLErrorCodeSQLExceptionTranslator.class);
		mapaDatabaseSqlErrorTranslators.put("Microsoft SQL Server", SQLServerSQLErrorCodeSQLExceptionTranslator.class);
		
		//dialects
		mapaDatabaseDialect.put("PostgreSQL", "org.hibernate.dialect.PostgreSQLDialect");
		mapaDatabaseDialect.put("Oracle", "org.hibernate.dialect.Oracle9Dialect");
		mapaDatabaseDialect.put("Microsoft SQL Server", "org.hibernate.dialect.SQLServerDialect");
		mapaDatabaseDialect.put("MySQL5", "org.hibernate.dialect.MySQL5Dialect");
		mapaDatabaseDialect.put("MySQL4", "org.hibernate.dialect.MySQLInnoDBDialect");
	}

	public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
		
		boolean containsDataSource = dataSourceConfigStrategy.configureDataSource(beanFactory);
		
		//procurar o atributo do config do neo
		verificarConfig(beanFactory);
		
		//autoregistrar beans do hibernate e banco de dados
		boolean containsSQLErrorCodeSQLExceptionTranslator = Util.beanFacotries.containsBeanOfClass(beanFactory, SQLErrorCodeSQLExceptionTranslator.class);
		boolean containsNeoImageResolver = Util.beanFacotries.containsBeanOfClass(beanFactory, NeoImageResolver.class);
		boolean containsNeoFormater = Util.beanFacotries.containsBeanOfClass(beanFactory, NeoFormater.class); 
		boolean containsAuthorizationDAO = Util.beanFacotries.containsBeanOfClass(beanFactory, AuthorizationDAO.class);
		boolean containsSessionFactory = Util.beanFacotries.containsBeanOfClass(beanFactory, SessionFactory.class);
		boolean containsJdbcTemplate = Util.beanFacotries.containsBeanOfClass(beanFactory, JdbcTemplate.class);
		boolean containsHibernateTemplate = Util.beanFacotries.containsBeanOfClass(beanFactory, HibernateTemplate.class);
		boolean containsPlatformTransactionManager = Util.beanFacotries.containsBeanOfClass(beanFactory, PlatformTransactionManager.class);
		boolean containsTransactionTemplate = Util.beanFacotries.containsBeanOfClass(beanFactory, TransactionTemplate.class);
		
		//TODO CONSERTAR O TRANSLATOR
		
		if(containsDataSource && !containsSQLErrorCodeSQLExceptionTranslator){
			String databaseName = getDatabaseName(beanFactory);
			Class class1 = mapaDatabaseSqlErrorTranslators.get(databaseName);
			if(class1 != null){
				MutablePropertyValues mpvs = new MutablePropertyValues();
				mpvs.addPropertyValue("dataSource", new RuntimeBeanReference(getDataSourceBeanName(beanFactory)));
				Util.beanFacotries.registerBean(beanFactory, class1, "sqlErrorCodesTranslator", mpvs, false);
				log.info("Registrando SQLErrorCodesTranslator .. "+class1.getSimpleName()+" para banco de dados "+databaseName);
				containsSQLErrorCodeSQLExceptionTranslator = true;
				
				// fazer o autowire dos beans que precisam do SQLErrorCodesTranslator
				autowireBean(beanFactory, HibernateTransactionManager.class);
				autowireBean(beanFactory, HibernateTemplate.class);
			}
		}
		if(!containsNeoImageResolver){
			Util.beanFacotries.registerBean(beanFactory, NeoImageResolver.class);
		}
		if(!containsNeoFormater){
			Util.beanFacotries.registerBean(beanFactory, NeoFormater.class);
		}
		if(!containsAuthorizationDAO){
			configureAuthorizationDAO(beanFactory);
		}
		if(containsDataSource && !containsJdbcTemplate){
			Util.beanFacotries.registerBean(beanFactory, JdbcTemplate.class);
			containsJdbcTemplate = true;
		}
		if(containsDataSource && !containsSessionFactory){
			String databaseName = getDatabaseName(beanFactory);
			
			Properties properties = new Properties();
			properties.setProperty("hibernate.dialect", mapaDatabaseDialect.get(databaseName));
			properties.setProperty("hibernate.show_sql", "true");
			
			MutablePropertyValues mutablePropertyValues = new MutablePropertyValues();
			mutablePropertyValues.addPropertyValue("hibernateProperties", properties);
			mutablePropertyValues.addPropertyValue("dataSource", new RuntimeBeanReference(getDataSourceBeanName(beanFactory)));
			
			Util.beanFacotries.registerBean(beanFactory, AnnotationSessionFactoryBean.class, "sessionFactory", mutablePropertyValues, false);
			containsSessionFactory = true;
		}
		if(containsSessionFactory && !containsHibernateTemplate){
			Util.beanFacotries.registerBean(beanFactory, HibernateTemplate.class);
			containsHibernateTemplate = true;
		}
		if(containsDataSource && containsSessionFactory && !containsPlatformTransactionManager){
			Util.beanFacotries.registerBean(beanFactory, HibernateTransactionManager.class);
			containsPlatformTransactionManager = true;
		}
		if(containsPlatformTransactionManager && !containsTransactionTemplate){
			Util.beanFacotries.registerBean(beanFactory, TransactionTemplate.class);
		}
	
		//se o config est no xml precisamos forar o bind do autorizationao que pode no estar no XML
		if(configInXML && authorizationDAOAutoconfig){
			//esse autowire forado  por causa do autorizationDao
			beanFactory.autowireBeanProperties(getConfig(beanFactory), AutowireCapableBeanFactory.AUTOWIRE_BY_TYPE, false);
		}
		if(!authorizationDAOAutoconfig){
			autowireBean(beanFactory, AuthorizationDAO.class);	
		}
		
		try {
			Class.forName("javax.servlet.http.HttpServletRequest");

			//se existe HttpServletRequest inicializar o uploader de arquivo
			MutablePropertyValues mutablePropertyValues = new MutablePropertyValues();
			mutablePropertyValues.addPropertyValue("maxUploadSize", getConfig(beanFactory).getMaxUploadSize()); 
			Util.beanFacotries.registerBean(beanFactory, NeoCommonsMultipartResolver.class, DispatcherServlet.MULTIPART_RESOLVER_BEAN_NAME, mutablePropertyValues);
		} catch (ClassNotFoundException e) {
			
		}
		
		
	}

	@SuppressWarnings("unchecked")
	private String getDataSourceBeanName(ConfigurableListableBeanFactory beanFactory) {
		Map map = beanFactory.getBeansOfType(DataSource.class);
		String dataSourcePropertyName = (String) map.keySet().iterator().next();
		return dataSourcePropertyName;
	}

	private String getDatabaseName(ConfigurableListableBeanFactory beanFactory) {
		DataSource dataSource = getDataSource(beanFactory);
		if(dataSource != null){
			Connection connection = null;
			DatabaseMetaData metaData = null;
			try {
				connection = dataSource.getConnection();	
				metaData = connection.getMetaData();
				
				String databaseProductName = metaData.getDatabaseProductName();
				//Para MySQL precisamos da verso para saber o dialeto TODO melhorar a forma de detectar a versao
				if("MySQL".equals(databaseProductName)){
					int databaseMajorVersion = metaData.getDatabaseMajorVersion();
					databaseProductName += databaseMajorVersion;
				}
				
				return databaseProductName;
			} catch (SQLException e) {
				String extra = " ";
				if(connection == null){
					extra = "No foi possvel estabelecer conexo com o banco de dados. ";
				}
				throw new NeoException("Erro ao tentar adquirir o nome do banco de dados. "+extra+e.getMessage(), e);
			} finally {
				try {
					if(metaData != null) connection.close();
					if(connection != null && !connection.isClosed()) connection.close();
				} catch (SQLException e) {
					e.printStackTrace();
				}
			}
		}
		return null;
	}

	private void configureAuthorizationDAO(ConfigurableListableBeanFactory beanFactory) {
		//procurar uma classe authorizationDAO na aplicao
		Class<?>[] allClassesOfType = classManager.getAllClassesOfType(AuthorizationDAO.class);
		if(allClassesOfType.length == 1){
			//achamos uma classe que pode ser o autorizationDAO
			log.info("AuthorizationDAO no configurado no XML mas encontrado na aplicao. Utilizando classe: "+allClassesOfType[0].getName());
			Util.beanFacotries.registerBean(beanFactory, allClassesOfType[0], "authorizationDAO");
			if(configInXML){
				//se o config foi encontrado no XML precisamos forar o autowire para detectar o authorizationDAO
				autowireConfig(beanFactory);
			}
			authorizationDAOAutoconfig = true;
		} else if(allClassesOfType.length > 1){
			throw new NeoException("Nenhum authorizationDAO foi configurado no XML mas foram encontrados mais de uma classe que implementa AuthorizationDAO na aplicao. " +
					"No  possvel escolher qual classe utilizar. Configure uma das classes no XML de configurao ou mantenha apenas uma classe implementando AuthorizationDAO na aplicao");
		}
	}

	private void autowireConfig(ConfigurableListableBeanFactory beanFactory) {
		AbstractBeanDefinition configBean = (AbstractBeanDefinition) beanFactory.getBeanDefinition(configName);
		configBean.setAutowireMode(AutowireCapableBeanFactory.AUTOWIRE_BY_TYPE);
	}
	
	private <E> void autowireBean(ConfigurableListableBeanFactory beanFactory, Class<E> class1) {
		E e = getBean(beanFactory, class1);
		if (e != null) {
			beanFactory.autowireBeanProperties(e, AutowireCapableBeanFactory.AUTOWIRE_BY_TYPE, false);
		}
	}

	/**
	 * Retorna o DataSource configurado na factory
	 * @param beanFactory
	 * @return
	 */
	private DataSource getDataSource(ConfigurableListableBeanFactory beanFactory){
		Map map = beanFactory.getBeansOfType(DataSource.class);
		if(map.size() > 1){
			log.warn("ATENO: Foi configurado mais de um DataSource na aplicao. Em alguns locais o NEO pode " +
					"precisar de algumas informaes do DataSource e com mais de um DataSource no  possvel determinar " +
					"qual  o DataSource correto. Isso pode levar a problemas na aplicao como a escolha de um dialeto incorreto");
		} 
		if(map.size() >= 1){
			@SuppressWarnings("unchecked")
			DataSource next = (DataSource) map.values().iterator().next();
			return next;
		} else {
			log.warn("Nenhum dataSource configurado na aplicao");
			return null;
		}
	} 
	
	/**
	 * Retorna o config configurado nesse facory
	 * @param beanFactory
	 * @return
	 */
	private Config getConfig(ConfigurableListableBeanFactory beanFactory) {
		return getBean(beanFactory, Config.class);
	}

	private <E> E getBean(ConfigurableListableBeanFactory beanFactory, Class<E> class1) {
		@SuppressWarnings("unchecked")
		Iterator iterator = beanFactory.getBeansOfType(class1).values().iterator();
		if(iterator.hasNext()){
			@SuppressWarnings("unchecked")
			E config = (E) iterator.next();
			return config;	
		} else {
			return null;
		}
	}


	private void verificarConfig(ConfigurableListableBeanFactory beanFactory) {
		Map<?, ?> map = beanFactory.getBeansOfType(Config.class);
		if(map.size() > 1){
			throw new ConfigurationException("Foram registrados mais de um bean no spring do tipo Config, s  possvel especificar um bean desse tipo");
		}
		if(map.isEmpty()){
			log.info("Config no encontrado. Utilizando default.");
			registerConfig(beanFactory);
		} else {
			Object nomeBean = map.keySet().iterator().next();
			configInXML = true;
			configName = (String) nomeBean;
			log.info("Encontrado Config. Utilizando bean "+nomeBean);
		}
	}
	
	
	private void registerConfig(ConfigurableListableBeanFactory beanFactory) {
		MutablePropertyValues mutablePropertyValues = new MutablePropertyValues();
		RootBeanDefinition beanDefinition = new RootBeanDefinition(DefaultConfig.class, mutablePropertyValues);
		beanDefinition.setAutowireMode(DefaultListableBeanFactory.AUTOWIRE_BY_TYPE);
		((DefaultListableBeanFactory)beanFactory).registerBeanDefinition("config", beanDefinition);	
	}

}
