/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.core.web.init;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import javax.servlet.ServletContext;
import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.MutablePropertyValues;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.RootBeanDefinition;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextException;
import org.springframework.jdbc.datasource.DriverManagerDataSource;
import org.springframework.jndi.JndiObjectFactoryBean;
import org.springframework.util.StringUtils;
import org.springframework.web.context.ConfigurableWebApplicationContext;
import org.springframework.web.context.WebApplicationContext;

import br.com.linkcom.neo.classmanager.ClassManager;
import br.com.linkcom.neo.classmanager.WebClassRegister;
import br.com.linkcom.neo.core.config.Config;
import br.com.linkcom.neo.core.web.NeoWeb;
import br.com.linkcom.neo.exception.ConfigurationException;
import br.com.linkcom.neo.exception.CouldNotCreateDataSourceException;
import br.com.linkcom.neo.exception.NeoException;
import br.com.linkcom.neo.util.Util;

/**
 * @author rogelgarcia
 * @since 21/01/2006
 * @version 1.1
 */
public class ContextLoader extends org.springframework.web.context.ContextLoader {

	protected static final Log log = LogFactory.getLog(ContextLoader.class);
	
	protected static final String CONFIG_LOCATION_PARAM = "configLocation";
	protected static final String DEFAULT_CONFIG_LOCATION = "/WEB-INF/applicationConfig.xml";
	protected static final Class NEO_DEFAULT_CONTEXT_CLASS = AnnotationsXmlWebApplicationContext.class;

	protected Config config;
	
	@Override
	protected WebApplicationContext createWebApplicationContext(ServletContext servletContext, ApplicationContext parent) throws BeansException {
		String contextClassName = servletContext.getInitParameter(CONTEXT_CLASS_PARAM);
		Class contextClass = NEO_DEFAULT_CONTEXT_CLASS; //LINHA MODIFICADA PARA UTILIZAR O CONTEXT CLASS DO NEO, QUE PROCURA OS BEANS COM ANNOTATIONS
		if (contextClassName != null) {
			try {
				contextClass = Class.forName(contextClassName, true, Thread.currentThread().getContextClassLoader());
			}
			catch (ClassNotFoundException ex) {
				throw new ApplicationContextException("Failed to load context class [" + contextClassName + "]", ex);
			}
			if (!ConfigurableWebApplicationContext.class.isAssignableFrom(contextClass)) {
				throw new ApplicationContextException("Custom context class [" + contextClassName +
						"] is not of type ConfigurableWebApplicationContext");
			}
		}

		ConfigurableWebApplicationContext wac =
				(ConfigurableWebApplicationContext) BeanUtils.instantiateClass(contextClass);
		wac.setParent(parent);
		wac.setServletContext(servletContext);
		if(AnnotationsXmlWebApplicationContext.class.isAssignableFrom(contextClass)){
			// o config j estar configurado pelo mtodo loadParentContext
			((AnnotationsXmlWebApplicationContext) wac).setBeanRegisters(config.getBeanRegisters());
			((AnnotationsXmlWebApplicationContext) wac).setTypeBeanRegisters(config.getTypeBeanRegisters());
		}
		String configLocation = servletContext.getInitParameter(CONFIG_LOCATION_PARAM);
		if (configLocation != null) {
			wac.setConfigLocations(StringUtils.tokenizeToStringArray(configLocation,
					ConfigurableWebApplicationContext.CONFIG_LOCATION_DELIMITERS));
		}

		wac.refresh();
		return wac;
	}

	@Override
	protected ApplicationContext loadParentContext(ServletContext servletContext) throws BeansException {
		//Config config = new DefaultConfig();
		//NeoWeb.createApplicationContext(servletContext, config);
		
		
		
		//cria um wac para tentar achar uma configurao personalizada do neo
		ConfigurableWebApplicationContext wac = new AnnotationsXmlWebApplicationContext();
		wac.setServletContext(servletContext);
		
		String configLocation = servletContext.getInitParameter(CONFIG_LOCATION_PARAM);
		if(configLocation == null){
			configLocation = DEFAULT_CONFIG_LOCATION;
		}
		URL resource;
		try {
			resource = servletContext.getResource(configLocation);
		} catch (MalformedURLException e) {
			throw new ConfigurationException("Caminho invlido do arquivo de configurao: "+configLocation, e);
		}
		if (resource != null) {
		
			wac.setConfigLocations(StringUtils.tokenizeToStringArray(configLocation,
					ConfigurableWebApplicationContext.CONFIG_LOCATION_DELIMITERS));			
	
		}
		ClassManager classManager;
		try {
			classManager = WebClassRegister.getClassManager(servletContext, "br.com.linkcom.neo");
		} catch (IOException e) {
			throw new NeoException("No foi possvel inicializar o contexto NEO ", e);
		}
		
		//TODO CRIAR O DATASOURCESTRATEGY
		NeoBeanFactoryPostProcessor neoBeanFactoryPostProcessor = new NeoBeanFactoryPostProcessor(classManager, new WebDataSourceConfigStrategy());
		wac.addBeanFactoryPostProcessor(neoBeanFactoryPostProcessor);
		wac.refresh();
		

		Map<?, ?> map = wac.getBeansOfType(Config.class);
		Object nomeBean = map.keySet().iterator().next();
		this.config = (Config) map.get(nomeBean);

	
		//INICIALIZAR CONTEXTO DE APLICAO NEO
		log.info("Criando contexto de aplicao Neo");
		NeoWeb.createApplicationContext(servletContext, config);

		config.init();
		
		return wac;
		
	}

}

class WebDataSourceConfigStrategy implements DataSourceConfigStrategy {

	Log log = LogFactory.getLog(WebDataSourceConfigStrategy.class);
	
	private DataSourceInfo dataSourceInfo;
	boolean fromResource = false;

	public WebDataSourceConfigStrategy(){

	}

	private void init() {
		ResourceBundle bundle = null;
		try {
			bundle = ResourceBundle.getBundle("connection");
		} catch (MissingResourceException e) {
			// caso nao encontre o bundle nao fazer nada
			
		}
		if (bundle != null) {
			try {
				//driver, url, username, password
				String jndi = null;
				try {
					jndi = bundle.getString("jndi");
				} catch (MissingResourceException e) {
					
				}
				String driver = null;
				String url = null;
				String username = null;
				String password = null;
				if (jndi == null) {
					driver = bundle.getString("driver");
					url = bundle.getString("url");
					try {
						username = bundle.getString("username");
						password = bundle.getString("password");
					} catch (Exception e) {
						//o nome de usurio e o password sao opcionais
					}
				}
				this.dataSourceInfo = new DataSourceInfo(jndi, driver, url, username, password);

			} catch (MissingResourceException e) {
				throw new NeoException("Erro ao carregar informaes de connection.properties. O arquivo est incorreto, " +
						"faltando algum dos parametros: driver, url, jndi");
			} 
		}
	}

	public boolean configureDataSource(ConfigurableListableBeanFactory beanFactory) throws CouldNotCreateDataSourceException {
		String dataSourceBeanName = getDataSourceBeanName(beanFactory);
		boolean containsDataSource = dataSourceBeanName != null;

		if(!containsDataSource){
			init();
			if (dataSourceInfo != null) {
				final boolean jndi = dataSourceInfo.jndi != null;
				log.info("Criando dataSource com informaes do arquivo connection.properties. Utilizando: " + (jndi ? " jndi " : " driver, url"));

				final DriverManagerDataSource dmds = new DriverManagerDataSource();
				if (!jndi) {
					dmds.setUrl(dataSourceInfo.url);
					dmds.setDriverClassName(dataSourceInfo.driver);
					dmds.setUsername(dataSourceInfo.username);
					dmds.setPassword(dataSourceInfo.password);
				}

				dataSourceBeanName = dataSourceBeanName != null ? dataSourceBeanName : "dataSource";
				FactoryBean factoryBean = new FactoryBean() {

					public Object getObject() throws Exception {
						return dmds;
					}

					public Class getObjectType() {
						return DataSource.class;
					}

					public boolean isSingleton() {
						return true;
					}
				};
				if(!jndi){
					beanFactory.registerSingleton(dataSourceBeanName, factoryBean);
				} else {
					MutablePropertyValues propertyValues = new MutablePropertyValues();
					propertyValues.addPropertyValue("jndiName", dataSourceInfo.jndi);
					//propertyValues.addPropertyValue("proxyInterface", DataSource.class);
					///propertyValues.addPropertyValue("cache", false);
					Util.beanFacotries.registerBean(beanFactory, JndiObjectFactoryBean.class, "dataSource", propertyValues);
				}
				containsDataSource = true;
			}
		}
		return containsDataSource;
	}

	private String getDataSourceBeanName(ConfigurableListableBeanFactory beanFactory) {
		String[] beanDefinitionNames = beanFactory.getBeanDefinitionNames();
		for (String beanDefinitionName : beanDefinitionNames) {
			BeanDefinition beanDefinition = beanFactory.getBeanDefinition(beanDefinitionName);
			if(beanDefinition instanceof RootBeanDefinition){
				String beanClassName = ((RootBeanDefinition)beanDefinition).getBeanClassName();
				try {
					Class<?> forName = Class.forName(beanClassName);
					if(DataSource.class.isAssignableFrom(forName)){
						return beanDefinitionName;
					} else if(FactoryBean.class.isAssignableFrom(forName)){
						if(JndiObjectFactoryBean.class.isAssignableFrom(forName) && beanDefinitionName.equals("dataSource")){
							//se tiver um bean JNDIObjectFactoryBean e ele se chamar dataSource consideraremos ele como dataSource
							return beanDefinitionName;
						}
					}
				} catch (ClassNotFoundException e) {
					throw new NeoException("O bean configurado com o nome "+beanDefinitionName+" est " +
							"configurado com uma classe inexistente",e);
					
				}
			}
		}
		return null;
	}
	
	
}
class DataSourceInfo{
	String jndi;
	String driver;
	String url;
	String username;
	String password;

	public DataSourceInfo(String jndi, String driver, String url, String username, String password){
		this.jndi = jndi;
		this.driver = driver;
		this.url = url;
		this.username = username;
		this.password = password;
	}
}
