/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.core.web.init;

import java.io.IOException;
import java.lang.annotation.Annotation;
import java.net.MalformedURLException;
import java.util.Arrays;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.web.context.support.XmlWebApplicationContext;

import br.com.linkcom.neo.bean.AnnotatedBeanRegister;
import br.com.linkcom.neo.bean.TypeBeanRegister;
import br.com.linkcom.neo.classmanager.ClassManager;
import br.com.linkcom.neo.classmanager.WebClassRegister;
import br.com.linkcom.neo.core.standard.Neo;


/**
 * ApplicationContext que registra beans no spring atravs de annotations
 * (No confunda com o ApplicationContext do pacote 
 * br.com.linkcom.neo.core.ApplicationContext)
 * @author rogelgarcia
 * @since 21/01/2006
 * @version 1.1
 */
public class AnnotationsXmlWebApplicationContext extends XmlWebApplicationContext {

	protected static final Log log = LogFactory.getLog(AnnotationsXmlWebApplicationContext.class);
	
	protected AnnotatedBeanRegister[] beanRegisters = new AnnotatedBeanRegister[0];

	private TypeBeanRegister[] typeBeanRegisters = new TypeBeanRegister[0];
	
	/**
	 * Configura esse application context com os registradores de bean
	 * @param beanRegisters
	 */
	public void setBeanRegisters(AnnotatedBeanRegister[] beanRegisters) {
		this.beanRegisters = beanRegisters;
	}

	@Override
	protected String[] getDefaultConfigLocations() {
		try {
			if(getServletContext().getResource(XmlWebApplicationContext.DEFAULT_CONFIG_LOCATION) == null){
				//log.warn("O arquivo de configurao default no foi encontrado: "+XmlWebApplicationContext.DEFAULT_CONFIG_LOCATION);
				return new String[]{};
			}
		} catch (MalformedURLException e) {
			//nao acontecer excecao
		}
		return super.getDefaultConfigLocations();
	}

	@Override
	protected void loadBeanDefinitions(DefaultListableBeanFactory beanFactory) throws IOException {
		
		// carregar primeiro as classe anotadas...
		// as classes no XML podero sobrescrever as anotadas
		ClassManager classManager = WebClassRegister.getClassManager(getServletContext(), "br.com.linkcom.neo");
		
		for (AnnotatedBeanRegister register : beanRegisters) {
			Class<? extends Annotation> annotationClass = register.getAnnotationClass();
			Class<?>[] classesWithAnnotation = classManager.getClassesWithAnnotation(annotationClass);
			register.registerBeans(Arrays.asList(classesWithAnnotation), beanFactory);
		}
		for (TypeBeanRegister register : typeBeanRegisters) {
			Class beanClass = register.getBeanClass();
			Class<?>[] allClassesOfType = classManager.getAllClassesOfType(beanClass);
			register.registerBeans(Arrays.asList(allClassesOfType), beanFactory);
		}
		if (Neo.isInApplicationContext()) {
			Neo.getApplicationContext().getConfig().setDefaultListableBeanFactory(beanFactory);
		}
		super.loadBeanDefinitions(beanFactory);
	}


	public void setTypeBeanRegisters(TypeBeanRegister[] typeBeanRegisters) {
		this.typeBeanRegisters  = typeBeanRegisters;
	}
}
