/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.core.web;

import java.security.Principal;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.springframework.validation.BindException;

import br.com.linkcom.neo.authorization.User;
import br.com.linkcom.neo.controller.Message;
import br.com.linkcom.neo.controller.MessageType;
import br.com.linkcom.neo.controller.MultiActionController;

/**
 * @author rogelgarcia
 * @since 21/01/2006
 * @version 1.1
 */
public class DefaultWebRequestContext implements WebRequestContext {
	
	String USER_ATTRIBUTE = "USER";

	String lastAction;
	HttpServletRequest httpServletRequest;
	HttpServletResponse httpServletResponse;
	String requestQuery;
	
	List<Message> messages = null;
	
	BindException bindException = new BindException(new Object(), "");

	private String firstRequestUrl;

	private WebApplicationContext applicationContext;
	
	public DefaultWebRequestContext(HttpServletRequest request, HttpServletResponse response, WebApplicationContext applicationContext) {
		super();
		httpServletRequest = request;
		httpServletResponse = response;
		this.applicationContext = applicationContext;
		String pathInfo = httpServletRequest.getPathInfo();
		if(pathInfo == null){
			pathInfo = "";
		}
		firstRequestUrl = httpServletRequest.getServletPath()+pathInfo;
		messages = getSessionMessages();
		getRequestQuery();//inicializar o request query
	}
	
	private List<Message> getSessionMessages() {
		HttpSession session = httpServletRequest.getSession();
		@SuppressWarnings("unchecked")
		List<Message> attribute = (List<Message>)session.getAttribute("_MESSAGES");
		if(attribute == null){
			attribute = new ArrayList<Message>();
			session.setAttribute("_MESSAGES", attribute);
		}
		return attribute;
	}

	public String getRequestQuery() {
		if(requestQuery == null){
			requestQuery = httpServletRequest.getServletPath() + (httpServletRequest.getPathInfo() != null ? httpServletRequest.getPathInfo() : "");
		}
		return requestQuery;
	}

	public String getParameter(String parameter) {
		return httpServletRequest.getParameter(parameter);
	}

	public void setAttribute(String name, Object value) {
		httpServletRequest.setAttribute(name, value);
	}

	public Principal getUserPrincipal() {
		return httpServletRequest.getUserPrincipal();
	}

	public User getUser() {
		return (User) httpServletRequest.getSession().getAttribute(USER_ATTRIBUTE);
	}

	public void setUser(User user) {
		httpServletRequest.getSession().setAttribute(USER_ATTRIBUTE, user);
	}

	public boolean hasRole(String role) {
		return httpServletRequest.getSession().getAttribute("PAPEL_" + role.toUpperCase()) != null;
	}

	public HttpServletRequest getServletRequest() {
		return httpServletRequest;
	}

	public HttpServletResponse getServletResponse() {
		return httpServletResponse;
	}

	public Object getAttribute(String name) {
		return httpServletRequest.getAttribute(name);
	}

	public String getFirstRequestUrl() {
		return firstRequestUrl;
	}

	public WebApplicationContext getWebApplicationContext() {
		return NeoWeb.getWebApplicationContext(httpServletRequest.getSession().getServletContext());
	}

	public HttpSession getSession() {
		return httpServletRequest.getSession();
	}

	public BindException getBindException() {
		return bindException;
	}

	public void setBindException(BindException bindException) {
		this.bindException = bindException;
	}

	public Message[] getMessages() {
		return messages.toArray(new Message[messages.size()]);
	}

	public void addMessage(Object source) {
		messages.add(new Message(MessageType.INFO, source));
	}
	
	public void addError(Object source) {
		messages.add(new Message(MessageType.ERROR, source));
	}

	public void clearMessages(){
		messages.clear();
	}
	
	public void addMessage(Object source, MessageType type) {
		messages.add(new Message(type, source));
	}

	public String getLastAction() {
		if(lastAction == null){
			lastAction = httpServletRequest.getParameter(MultiActionController.ACTION_PARAMETER);
		}
		return lastAction;
	}

	public void setLastAction(String lastAction) {
		this.lastAction = lastAction;
	}

	
	public WebApplicationContext getApplicationContext() {
		return applicationContext;
	}

	public Object getUserAttribute(String name) {
		return httpServletRequest.getSession().getAttribute(name);
	}

	public void setUserAttribute(String name, Object value) {
		httpServletRequest.getSession().setAttribute(name, value);
	}

	
	
}
