/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.core.standard;

import java.io.IOException;
import java.sql.SQLException;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.Properties;
import java.util.ResourceBundle;

import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.RootBeanDefinition;
import org.springframework.jdbc.datasource.DriverManagerDataSource;
import org.springframework.jndi.JndiObjectFactoryBean;

import br.com.linkcom.neo.authorization.User;
import br.com.linkcom.neo.classmanager.ClassRegister;
import br.com.linkcom.neo.core.config.Config;
import br.com.linkcom.neo.core.config.DefaultConfig;
import br.com.linkcom.neo.core.standard.init.AnnotationsXmlApplicationContext;
import br.com.linkcom.neo.core.web.init.DataSourceConfigStrategy;
import br.com.linkcom.neo.core.web.init.NeoBeanFactoryPostProcessor;
import br.com.linkcom.neo.exception.CouldNotCreateDataSourceException;
import br.com.linkcom.neo.exception.NeoException;

/**
 * @author rogelgarcia
 * @since 21/01/2006
 * @version 1.1
 */
public class NeoStandard extends Neo {
	
	private static DataSourceInfo dataSourceInfo;
	
	/**
	 * Configura o dataSource do neo utilizado para ambientes JSE.<BR>
	 * Caso tenha sido configurado algum dataSource no XML de configurao da aplicao, ele ser substituido 
	 * por um dataSource que utilize as configuraes fornecidas na chamada desse metdo.<BR>
	 * Essa configurao  util em casos de testes onde no exista um ambiente JNDI.<BR>
	 * Esse mtodo s causa efeito caso a aplicao esteje rodando em JSE.<BR>
	 * Esse mtodo deve ser chamado antes do mtodo createNeoContext(..).<BR>
	 * Caso o nico bean configurado no XML seje o dataSource no  necessrio informar o caminho arquivo de configurao 
	 * ao criar o contexto NEO (chamar o mtodo createNeoContext()) se for utilizado esse mtodo.<BR>
	 * O banco de dados tambm pode ser configurado atravs do arquivo de propriedades connection.properties.<BR>
	 * Caso exista o arquivo connection.properties a chamada a esse mtodo no  necessria (algum dataSource que tenha sido configurado 
	 * no XML ser substituido por um com as propriedade do connection.properties).<BR>
	 * O arquivo connection.properties deve conter as seguintes propriedades: driver, url, username, password
	 * O arquivo connection.properties tambm pode ser utilizado em ambientes JEE caso no tenha sido 
	 * configurado nenhum dataSource no XML de configurao. Caso o banco de dados deva ser configurado via JNDI  necessrio 
	 * especificar no arquivo a propriedade jndi com o caminho jndi do dataSource. Em ambientes JSE a propriedade jndi  ignorada 
	 * caso existam as outras propriedades (driver, url, username, password). Em ambientes JEE a propriedade jndi tem preferencia.
	 * @param driver Driver do banco de dados
	 * @param url Url do banco de dados
	 * @param username nome de usurio para conectar ao banco de dados
	 * @param password senha para conectar ao banco de dados
	 * @throws SQLException
	 */
	public static void configureDataSource(String driver, String url, String username, String password) {
		dataSourceInfo = new DataSourceInfo(driver, url, username, password);
	}
	
	public static void createNeoApplicationContext(Config config){
		//BasicConfigurator.configure(new ConsoleAppender(new SimpleLayout()));
		ApplicationContext applicationContext = new DefaultApplicationContext(config);
		Neo.applicationContext.set(applicationContext);
	}
	
	public static void createNeoApplicationContext(){
		createNeoApplicationContext(new DefaultConfig());
	}
	
	public static RequestContext createNeoContext(String... locations){
		//createNeoApplicationContext();
		//inicializar o log4J
		initLog4J();
		
		AnnotationsXmlApplicationContext xac = new AnnotationsXmlApplicationContext(locations);
		
		NeoBeanFactoryPostProcessor neoBeanFactoryPostProcessor = null;
		StandardDataSourceConfigStrategy configStrategy = new StandardDataSourceConfigStrategy(dataSourceInfo);
		try {
			neoBeanFactoryPostProcessor = new NeoBeanFactoryPostProcessor(ClassRegister.getClassManager(), configStrategy);
		} catch (IOException e) {
			throw new NeoException("No foi possvel inicializar o contexto NEO ",e);
		}
		xac.addBeanFactoryPostProcessor(neoBeanFactoryPostProcessor);
		xac.refresh();
		
		
		
		Map<?, ?> map = xac.getBeansOfType(Config.class);
		Object nomeBean = map.keySet().iterator().next();
		Config config = (Config) map.get(nomeBean);
		
		createNeoApplicationContext(config);
		
		config.init();
		
		xac.setBeanRegisters(config.getBeanRegisters());
		xac.setTypeBeanRegisters(config.getTypeBeanRegisters());
		
		xac.refresh();
		
//		//TODO CARREGAR CONFIGURAES DO XML DE CONFIGURACAO
//		
//		XmlBeanDefinitionReader reader = new XmlBeanDefinitionReader(parent);
//		
//		
//		//carregar primeiro as classe anotadas...
//		// as classes no XML podero sobrescrever as anotadas
//		br.com.linkcom.neo.bean.AnnotatedBeanRegister[] beanRegisters = Neo.getApplicationContext().getConfig().getBeanRegisters();
//		ClassManager classManager = Neo.getApplicationContext().getClassManager();
//		
//		for (AnnotatedBeanRegister register : beanRegisters) {
//			Class<? extends Annotation> annotationClass = register.getAnnotationClass();
//			Class<?>[] classesWithAnnotation = classManager.getClassesWithAnnotation(annotationClass);
//			register.registerBeans(Arrays.asList(classesWithAnnotation), parent);
//		}
//		if (Neo.isInApplicationContext()) {
//			Neo.getApplicationContext().getConfig().setDefaultListableBeanFactory(parent);
//		}
//		
//		for (Resource resource : resources) {
//			reader.loadBeanDefinitions(resource);
//		}
		
		DefaultRequestContext requestContext = new DefaultRequestContext(applicationContext.get());
		Neo.requestContext.set(requestContext);
		/* TODO SETAR O FACTORY
		DefaultMirageApplicationContext ac = (DefaultMirageApplicationContext) Mirage.getApplicationContext();
		ac.setFactory(beanFactory == null? parent: beanFactory);
		
		MirageRequestContext requestContext = new DefaultMirageRequestContext(getApplicationContext());
		*/
		
		
		return requestContext;
	}

	public static void initLog4J() {
		Properties properties = new Properties();
		properties.setProperty("log4j.defaultInitOverride", "false");
		properties.setProperty("log4j.rootCategory", "INFO, console");
		properties.setProperty("log4j.appender.console", "org.apache.log4j.ConsoleAppender");
		properties.setProperty("log4j.appender.console.layout", "org.apache.log4j.PatternLayout");
		properties.setProperty("log4j.appender.console.layout.ConversionPattern", "%-5p %c %x - %m%n");
//		PropertyConfigurator.configure(properties);
	}
	
	
	public static RequestContext createNeoContext(){
		return createNeoContext(new String[0]);
	}

	public static void setUser(User user) {
		DefaultRequestContext requestContext2 = (DefaultRequestContext)Neo.getRequestContext();
		requestContext2.setUser(user);
	}
	
}
class StandardDataSourceConfigStrategy implements DataSourceConfigStrategy {

	Log log = LogFactory.getLog(StandardDataSourceConfigStrategy.class);
	
	private DataSourceInfo dataSourceInfo;
	boolean fromResource = false;

	public StandardDataSourceConfigStrategy(DataSourceInfo dataSourceInfo){
		this.dataSourceInfo = dataSourceInfo;
		ResourceBundle bundle = null;
		try {
			bundle = ResourceBundle.getBundle("connection");
		} catch (MissingResourceException e) {
			// caso nao encontre o bundle nao fazer nada
			
		}
		if (bundle != null) {
			try {
				//driver, url, username, password
				String driver = bundle.getString("driver");
				String url = bundle.getString("url");
				String username = null;
				String password = null;
				try {
					username = bundle.getString("username");
					password = bundle.getString("password");
				} catch (Exception e) {
					//o nome de usurio e o password sao opcionais
				}
				this.dataSourceInfo = new DataSourceInfo(driver, url, username, password);
				fromResource = true;
			} catch (MissingResourceException e) {
				throw new NeoException("Erro ao carregar informaes de connection.properties. O arquivo est incorreto, " +
						"faltando algum dos parametros: driver, url");
			} 
		}	

	}

	public boolean configureDataSource(ConfigurableListableBeanFactory beanFactory) throws CouldNotCreateDataSourceException {
		String dataSourceBeanName = getDataSourceBeanName(beanFactory);
		boolean containsDataSource = dataSourceBeanName != null;
		if(fromResource){
			log.info("Criando dataSource com informaes do arquivo connection.properties");
		} else if (dataSourceInfo != null){
			log.info("Criando dataSource com informaes fornecidas pelo mtodo NeoStandard.configureDataSource(..)");
		}
		if(dataSourceInfo != null){
			
			final DriverManagerDataSource dataSource = new DriverManagerDataSource();
			dataSource.setUrl(dataSourceInfo.url);
			dataSource.setDriverClassName(dataSourceInfo.driver);
			dataSource.setUsername(dataSourceInfo.username);
			dataSource.setPassword(dataSourceInfo.password);
			dataSourceBeanName = dataSourceBeanName != null? dataSourceBeanName : "dataSource";
			FactoryBean factoryBean = new FactoryBean(){

				public Object getObject() throws Exception {
					return dataSource;
				}

				public Class getObjectType() {
					return DataSource.class;
				}

				public boolean isSingleton() {
					return true;
				}};
			beanFactory.registerSingleton(dataSourceBeanName, factoryBean);
			containsDataSource = true;
		}
		return containsDataSource;
	}

	private String getDataSourceBeanName(ConfigurableListableBeanFactory beanFactory) {
		String[] beanDefinitionNames = beanFactory.getBeanDefinitionNames();
		for (String beanDefinitionName : beanDefinitionNames) {
			BeanDefinition beanDefinition = beanFactory.getBeanDefinition(beanDefinitionName);
			if(beanDefinition instanceof RootBeanDefinition){
				String beanClassName = ((RootBeanDefinition)beanDefinition).getBeanClassName();
				try {
					Class<?> forName = Class.forName(beanClassName);
					if(DataSource.class.isAssignableFrom(forName)){
						return beanDefinitionName;
					} else if(FactoryBean.class.isAssignableFrom(forName)){
						if(JndiObjectFactoryBean.class.isAssignableFrom(forName) && beanDefinitionName.equals("dataSource")){
							//se tiver um bean JNDIObjectFactoryBean e ele se chamar dataSource consideraremos ele como dataSource
							return beanDefinitionName;
						}
					}
				} catch (ClassNotFoundException e) {
					throw new NeoException("O bean configurado com o nome "+beanDefinitionName+" est " +
							"configurado com uma classe inexistente",e);
					
				}
			}
		}
		return null;
	}
	
	
}
class DataSourceInfo{
	String driver;
	String url;
	String username;
	String password;

	public DataSourceInfo(String driver, String url, String username, String password){
		this.driver = driver;
		this.url = url;
		this.username = username;
		this.password = password;
	}
}
