/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.core.config;

import java.util.Properties;

import org.springframework.beans.factory.support.DefaultListableBeanFactory;

import br.com.linkcom.neo.authorization.AuthorizationDAO;
import br.com.linkcom.neo.authorization.AuthorizationManager;
import br.com.linkcom.neo.authorization.PermissionLocator;
import br.com.linkcom.neo.authorization.UserLocator;
import br.com.linkcom.neo.authorization.impl.AuthorizationDAOImpl;
import br.com.linkcom.neo.authorization.impl.AuthorizationManagerImpl;
import br.com.linkcom.neo.authorization.impl.ControlMappingLocatorImpl;
import br.com.linkcom.neo.authorization.impl.PermissionLocatorImpl;
import br.com.linkcom.neo.authorization.impl.UserLocatorImpl;
import br.com.linkcom.neo.bean.AnnotatedBeanRegister;
import br.com.linkcom.neo.bean.BeanDescriptorFactory;
import br.com.linkcom.neo.bean.BeanDescriptorFactoryImpl;
import br.com.linkcom.neo.bean.BeanRegisterBean;
import br.com.linkcom.neo.bean.BeanRegisterCrud;
import br.com.linkcom.neo.bean.BeanRegisterDao;
import br.com.linkcom.neo.bean.BeanRegisterService;
import br.com.linkcom.neo.bean.ControllerBeanRegister;
import br.com.linkcom.neo.bean.CrudBeanRegister;
import br.com.linkcom.neo.bean.DAOBeanRegister;
import br.com.linkcom.neo.bean.ServiceBeanRegister;
import br.com.linkcom.neo.bean.TypeBeanRegister;
import br.com.linkcom.neo.controller.ControlMappingLocator;
import br.com.linkcom.neo.core.standard.Neo;
import br.com.linkcom.neo.core.web.WebApplicationContext;
import br.com.linkcom.neo.exception.ReportException;
import br.com.linkcom.neo.report.ReportGenerator;
import br.com.linkcom.neo.report.ReportGeneratorImpl;
import br.com.linkcom.neo.report.ReportNameResolverImpl;
import br.com.linkcom.neo.report.ReportTranslatorImpl;
import br.com.linkcom.neo.rtf.RTFGenerator;
import br.com.linkcom.neo.rtf.RTFGeneratorImpl;
import br.com.linkcom.neo.rtf.RTFNameResolverImpl;
import br.com.linkcom.neo.validation.ValidatorRegistryImpl;

/**
 * @author rogelgarcia
 * @since 21/01/2006
 * @version 1.1
 */
public class DefaultConfig implements Config {
	
	public static final String AUTO_CONFIG_HIBERNATE = "autoConfigHibernate";
	public static final String HIBERNATE_DIALECT = "dialect";
	
	protected DefaultListableBeanFactory defaultListableBeanFactory;

	protected static final AnnotatedBeanRegister[] beanRegisters = new AnnotatedBeanRegister[] { new BeanRegisterBean(), new BeanRegisterCrud(), new BeanRegisterService(), new BeanRegisterDao() };
	protected static final TypeBeanRegister[] typeBeanRegisters = new TypeBeanRegister[] { 
		new ControllerBeanRegister(), new CrudBeanRegister(), new ServiceBeanRegister(), new DAOBeanRegister()
	};

	protected Properties properties = new Properties();

	protected ReportGenerator reportGenerator;
	protected RTFGenerator rtfGenerator;

	protected AuthorizationManager authorizationManager;

	protected UserLocator userLocator;

	protected PermissionLocator permissionLocator;

	protected ControlMappingLocator controlMappingLocator;

	protected AuthorizationDAO authorizationDAO;
	
	protected BeanDescriptorFactory beanDescriptorFactory;
	
	protected ValidatorRegistry validatorRegistry;
	
	protected String requiredRenderType = "asterisk";
	
	protected int maxUploadSize = 10000000;//10Mb
	
	public int getMaxUploadSize() {
		return maxUploadSize;
	}

	public void setMaxUploadSize(int maxUploadSize) {
		this.maxUploadSize = maxUploadSize;
	}

	public ValidatorRegistry getValidatorRegistry() {
		return validatorRegistry;
	}

	public void init() {
		if (authorizationDAO == null) {
			authorizationDAO = new AuthorizationDAOImpl();
		}
		if (controlMappingLocator == null) {
			controlMappingLocator = new ControlMappingLocatorImpl();
		}
		if (permissionLocator == null) {
			permissionLocator = new PermissionLocatorImpl();
			((PermissionLocatorImpl)permissionLocator).setAuthorizationDAO(authorizationDAO);
		}
		if (userLocator == null) {
			userLocator = new UserLocatorImpl();
			((UserLocatorImpl)userLocator).setAuthorizationDAO(authorizationDAO);
		}
		if (authorizationManager == null) {
			authorizationManager = new AuthorizationManagerImpl();
			((AuthorizationManagerImpl)authorizationManager).setControlMappingLocator(controlMappingLocator);
			((AuthorizationManagerImpl)authorizationManager).setPermissionLocator(permissionLocator);
			((AuthorizationManagerImpl)authorizationManager).setUserLocator(userLocator);
		}
		if(beanDescriptorFactory == null){
			beanDescriptorFactory = new BeanDescriptorFactoryImpl();
		}
		if(validatorRegistry == null){
			validatorRegistry = new ValidatorRegistryImpl();
		}
	}

	public AuthorizationDAO getAuthorizationDAO() {
		return authorizationDAO;
	}
	
	public void setAuthorizationDAO(AuthorizationDAO authorizationDAO) {
		this.authorizationDAO = authorizationDAO;
	}

	public AuthorizationManager getAuthorizationManager() {
		return authorizationManager;
	}

	public void setAuthorizationManager(AuthorizationManager authorizationManager) {
		this.authorizationManager = authorizationManager;
	}

	public ReportGenerator getReportGenerator() {
		if (reportGenerator == null) {
			try {
				// se nao estiver em um contexto Web d pau aqui
				WebApplicationContext applicationContext = (WebApplicationContext) Neo.getApplicationContext();
				ReportNameResolverImpl nameResolverImpl = new ReportNameResolverImpl("/WEB-INF/relatorio/", ".jasper", applicationContext.getServletContext());
				ReportTranslatorImpl translatorImpl = new ReportTranslatorImpl(nameResolverImpl);
				ReportGeneratorImpl generatorImpl = new ReportGeneratorImpl(translatorImpl);
				reportGenerator = generatorImpl;
			} catch (ClassCastException e) {
				throw new ReportException("A gerao de relatrios s pode ser feita em um contexto WEB");
			}
		}
		return reportGenerator;
	}
	
	public RTFGenerator getRTFGenerator(){
		if (rtfGenerator == null) {
			try {
				// se nao estiver em um contexto Web d pau aqui
				WebApplicationContext applicationContext = (WebApplicationContext) Neo.getApplicationContext();
				RTFNameResolverImpl nameResolverImpl = new RTFNameResolverImpl("/WEB-INF/rtf/", ".rtf", applicationContext.getServletContext());
				rtfGenerator = new RTFGeneratorImpl(nameResolverImpl);
			} catch (ClassCastException e) {
				throw new ReportException("A gerao de RTFs s pode ser feita em um contexto WEB");
			}			
		}

		return rtfGenerator;
	}

	public void setReportGenerator(ReportGenerator reportGenerator) {
		this.reportGenerator = reportGenerator;
	}

	public Properties getProperties() {
		return properties;
	}

	public void setProperties(Properties properties) {
		this.properties = properties;
	}

	public AnnotatedBeanRegister[] getBeanRegisters() {
		return beanRegisters;
	}

	public ControlMappingLocator getControlMappingLocator() {
		return controlMappingLocator;
	}

	public PermissionLocator getPermissionLocator() {
		return permissionLocator;
	}

	public UserLocator getUserLocator() {
		return userLocator;
	}

	public void setControlMappingLocator(ControlMappingLocator controlMappingLocator) {
		this.controlMappingLocator = controlMappingLocator;
	}

	public void setPermissionLocator(PermissionLocator permissionLocator) {
		this.permissionLocator = permissionLocator;
	}

	public void setUserLocator(UserLocator userLocator) {
		this.userLocator = userLocator;
	}

	public BeanDescriptorFactory getBeanDescriptorFactory() {
		return beanDescriptorFactory;
	}

	public DefaultListableBeanFactory getDefaultListableBeanFactory() {
		return defaultListableBeanFactory;
	}

	public void setDefaultListableBeanFactory(DefaultListableBeanFactory defaultListableBeanFactory) {
		this.defaultListableBeanFactory = defaultListableBeanFactory;
	}

	public TypeBeanRegister[] getTypeBeanRegisters() {
		return typeBeanRegisters;
	}
	
	public void setRequiredRenderType(String requiredRenderType) {
		this.requiredRenderType = requiredRenderType;
	}

	public String getRequiredRenderType() {
		return "asterisk";
	}
}
