/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.controller.crud;

import org.springframework.validation.BindException;
import org.springframework.web.servlet.ModelAndView;

import br.com.linkcom.neo.controller.Action;
import br.com.linkcom.neo.controller.Command;
import br.com.linkcom.neo.controller.DefaultAction;
import br.com.linkcom.neo.controller.Input;
import br.com.linkcom.neo.controller.MultiActionController;
import br.com.linkcom.neo.controller.OnErrors;
import br.com.linkcom.neo.core.standard.Neo;
import br.com.linkcom.neo.core.web.WebRequestContext;
import br.com.linkcom.neo.exception.NeoException;
import br.com.linkcom.neo.persistence.ListagemResult;
import br.com.linkcom.neo.service.GenericService;
import br.com.linkcom.neo.util.Util;

/**
 * @author Pedro Gonalves
 * @since 19/10/2007
 * @version 1.0
 */
public class SimpleCrudController<FORMBEAN,BEAN> extends MultiActionController {

	public static final String LISTAGEM = "listagem";

	public static final String ENTRADA = "entrada";

	public static final String CRIAR = "criar";

	public static final String EDITAR = "editar";
	
	public static final String CONSULTAR = "consultar";

	public static final String SALVAR = "salvar";

	public static final String EXCLUIR = "excluir";
	
	protected Class<BEAN> beanClass;
	protected Class<FORMBEAN> formClass;
	
	protected GenericService<BEAN> genericService;
	
	public GenericService<BEAN> getGenericService() {
		return genericService;
	}

	public void setGenericService(GenericService<BEAN> genericService) {
		this.genericService = genericService;
	}
	
	public void setBeanClass(Class<BEAN> beanClass) {
		this.beanClass = beanClass;
	}
	
	public void setFormClass(Class<FORMBEAN> formClass) {
		this.formClass = formClass;
	}
	
	@SuppressWarnings("unchecked")
	public SimpleCrudController(){
		Class[] classes =  Util.generics.getGenericTypes(this.getClass());
		if(classes.length != 2){
			classes = Util.generics.getGenericTypes2(this.getClass());
			if(classes.length == 2){
				setFormClass(classes[0]);
				setBeanClass(classes[1]);
				return;
			}

			throw new RuntimeException("A classe "+this.getClass().getName()+" deve declarar tres tipos genricos LISTAGEM, FORMBEAN, BEAN");
		}
		setFormClass(classes[0]);
		setBeanClass(classes[1]);
	}
	
	@Action(LISTAGEM)
	@Command(validate = true)
	@Input(LISTAGEM)
	@DefaultAction
	public ModelAndView handleRequest(WebRequestContext request,FORMBEAN bean) throws CrudException {
		setInfosTemplate(request, bean);
		setAuthotizationMethod(request,true);
		return getModelAndView(request,bean);
	}
	
	@Action(EDITAR)
	@Command(validate = true)
	@Input(LISTAGEM)
	@OnErrors(LISTAGEM)
	public ModelAndView doEditar(WebRequestContext request, FORMBEAN form) throws CrudException {
		try {
			BEAN bean = formToBean(form);
			bean = carregar(request, bean);
			form = beanToForm(bean);
		} catch (Exception e) {
			throw new CrudException(EDITAR, e);
		}
		setAuthotizationMethod(request,false);
		setInfosTemplate(request,form);
		return getModelAndView(request, form);
	}
	
	@Action(CONSULTAR)
	@Command(validate = true)
	@Input(LISTAGEM)
	@OnErrors(LISTAGEM)
	public ModelAndView doConsultar(WebRequestContext request, FORMBEAN form) throws CrudException {
		request.setAttribute(CONSULTAR, true);
		return doEditar(request, form);
	}
	
	@Action(EXCLUIR)
	@Command(validate = true)
	@Input(LISTAGEM)
	@OnErrors(LISTAGEM)
	public ModelAndView doExcluir(WebRequestContext request, FORMBEAN form) throws CrudException {
		BEAN bean = null; 
		try {
			bean = formToBean(form);
			excluir(request, bean);
		} catch (Exception e) {
			throw new CrudException(EXCLUIR, e);
		}
		return sendRedirect(request);
	}
	
	@Action(SALVAR)
	@Command(validate = true)
	@Input(LISTAGEM)
	@OnErrors(LISTAGEM)
	public ModelAndView doSalvar(WebRequestContext request, FORMBEAN form) throws CrudException{
		BEAN bean = null;
		try {
			bean = formToBean(form);
			salvar(request, bean);
		} catch (Exception e) {
			throw new CrudException(SALVAR, e);
		}
		return sendRedirect(request);
	}
	
	/**
	 * Envia para a tela o objeto com o nome do bean parametrizado no generics.
	 * @param request
	 * @param form
	 * @throws CrudException
	 */
	protected void setEntradaDefaultInfo(WebRequestContext request, FORMBEAN form) throws CrudException {
		request.setAttribute(getBeanName(), form);		
	}
	
	/**
	 * Envia as informaes para o template de entrada
	 * @param request
	 * @param form
	 */
	protected void setInfoForTemplate(WebRequestContext request, FORMBEAN form) {
		request.setAttribute("TEMPLATE_beanNameUncaptalized", getBeanName());
		request.setAttribute("TEMPLATE_beanName", getBeanName());
		request.setAttribute("TEMPLATE_enviar", "salvar");
		request.setAttribute("TEMPLATE_voltar", "listagem");
		request.setAttribute("TEMPLATE_beanDisplayName", Neo.getApplicationContext().getBeanDescriptor(null, beanClass).getDisplayName());
		request.setAttribute("TEMPLATE_beanClass", beanClass);
	}
	
	/**
	 * Seta as informaes default para o template principal
	 * @param request
	 * @param bean
	 * @throws CrudException
	 */
	protected void setInfosTemplate(WebRequestContext request,FORMBEAN bean) throws CrudException{
		setListagemInfo(request);
		setInfoForTemplate(request, bean);
		setEntradaDefaultInfo(request, bean);
		request.setAttribute("TEMPLATE_listagem", "true");
	}
	
	/**
	 * Envia os dados para o template de listagem
	 * @param request
	 * @throws CrudException
	 */
	protected void setListagemInfo(WebRequestContext request) throws CrudException {
		String attribute = request.getParameter("currentPage");
		Integer currentPage = 0;
		if(attribute != null)
			currentPage = Integer.parseInt(attribute);
		
		FiltroListagem filtroListagem = new FiltroListagem();
		filtroListagem.pageSize = 10;
		filtroListagem.setCurrentPage(currentPage);
		ListagemResult<BEAN> listagemResult = getLista(request,filtroListagem);
		request.setAttribute("lista", listagemResult.list());
		request.setAttribute("currentPage", filtroListagem.getCurrentPage());
		request.setAttribute("numberOfPages", filtroListagem.getNumberOfPages());
	}
	
	/**
	 * Prepara a lista para exibio na listagem de dados
	 * @param request
	 * @param filtro
	 * @return
	 */
	protected ListagemResult<BEAN> getLista(WebRequestContext request,FiltroListagem filtro) {
		return genericService.findForListagem(filtro);
	}
	
	/**
	 * Exclui o objeto
	 * @param request
	 * @param bean
	 * @throws Exception
	 */
	protected void excluir(WebRequestContext request, BEAN bean) throws Exception {
		genericService.delete(bean);
	}
	
	/**
	 * Carrega o objeto.
	 * @param request
	 * @param bean
	 * @return
	 * @throws Exception
	 */
	protected BEAN carregar(WebRequestContext request, BEAN bean) throws Exception {
		return genericService.loadForEntrada(bean);
	}
	
	/**
	 * Salva o objeto
	 * @param request
	 * @param bean
	 * @throws Exception
	 */
	protected void salvar(WebRequestContext request, BEAN bean) throws Exception {
		genericService.saveOrUpdate(bean);
	}
	
	/**
	 * Converte o bean para form
	 * @param bean
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public FORMBEAN beanToForm(BEAN bean){
		if(formClass.equals(beanClass)){
			return (FORMBEAN)bean;
		}
		throw new NeoException("No foi possvel converter de bean para form. Sobrescrever o mtodo beanToForm");
	}
	
	/**
	 * Converte o formulrio para um bean
	 * @param form
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public BEAN formToBean(FORMBEAN form){
		if(beanClass.equals(formClass)){
			return (BEAN)form;
		}
		throw new NeoException("No foi possvel converter de form para bean. Sobrescrever o mtodo formToBean");
	}

	/**
	 * Mtodo responsvel por redirecionar quando executa as aes de excluir e editar
	 * @param request
	 * @param bean
	 * @return
	 */
	protected ModelAndView sendRedirect(WebRequestContext request) {
		return sendRedirectToAction("listagem");
	}

	/**
	 * Seta o jsp que ser exibido. Valor default: Primeira letra do Nome da classe do bean minscula e com o sufixo ".jsp".
	 * @param request
	 * @param bean
	 * @return
	 */
	protected ModelAndView getModelAndView(WebRequestContext request, FORMBEAN bean) {
		return new ModelAndView("crud/"+getBeanName()+"");
	}
	
	public String getBeanName() {
		return Util.strings.uncaptalize(beanClass.getSimpleName());
	}
	 
	public void setAuthotizationMethod(WebRequestContext context, Boolean type){
		if(type)
			context.setAttribute("TEMPLATE_authorization", "criar");
		else
			context.setAttribute("TEMPLATE_authorization", "editar");
	}
	
	protected void validateBean(FORMBEAN bean, BindException errors) {
		
	}
}
