/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.controller.crud;

import java.util.ArrayList;

import javax.servlet.http.HttpServletRequest;

import org.springframework.validation.BindException;
import org.springframework.web.servlet.ModelAndView;

import br.com.linkcom.neo.core.standard.Neo;
import br.com.linkcom.neo.core.web.DefaultWebRequestContext;
import br.com.linkcom.neo.core.web.WebRequestContext;
import br.com.linkcom.neo.exception.NeoException;
import br.com.linkcom.neo.persistence.ListagemResult;
import br.com.linkcom.neo.service.GenericService;
import br.com.linkcom.neo.util.Util;
import br.com.linkcom.neo.view.ajax.View;

/**
 * @author rogelgarcia
 * @since 01/02/2006
 * @version 1.1
 */
public class CrudController<FILTRO extends FiltroListagem, FORMBEAN, BEAN> extends AbstractCrudController<FILTRO, FORMBEAN>{
	
	protected Class<? extends FILTRO> listagemCommandClass;
	protected Class<FORMBEAN> entradaCommandClass;
	protected Class<BEAN> beanClass;
	
	protected GenericService<BEAN> genericService;
	
	public GenericService<BEAN> getGenericService() {
		return genericService;
	}

	public void setGenericService(GenericService<BEAN> genericService) {
		this.genericService = genericService;
	}
	
	public void setBeanClass(Class<BEAN> beanClass) {
		this.beanClass = beanClass;
	}

	public void setEntradaCommandClass(Class<FORMBEAN> entradaCommandClass) {
		this.entradaCommandClass = entradaCommandClass;
	}

	public void setListagemCommandClass(Class<FILTRO> listagemCommandClass) {
		this.listagemCommandClass = listagemCommandClass;
	}

	@SuppressWarnings("unchecked")
	@Override
	protected void validate(Object obj, BindException errors, String acao) {
		if(obj != null){
			if(listagemCommandClass.isAssignableFrom(obj.getClass())){
				validateFilter((FILTRO)obj, errors);
			}
			if(entradaCommandClass.isAssignableFrom(obj.getClass())){
				validateBean((FORMBEAN)obj, errors);
			}
		}
	}
	
	protected void validateBean(FORMBEAN bean, BindException errors) {
		
	}

	protected void validateFilter(FILTRO filtro, BindException errors) {
		
	}

	@SuppressWarnings("unchecked")
	public CrudController(){
		Class[] classes =  Util.generics.getGenericTypes(this.getClass());
		if(classes.length != 3){
			//tentar a outra forma de Generics
			{
				classes = Util.generics.getGenericTypes2(this.getClass());
				if(classes.length == 3 && !classes[1].equals(Object.class) && !classes[2].equals(Object.class)){
					setListagemCommandClass(classes[0]);
					setEntradaCommandClass(classes[1]);
					setBeanClass(classes[2]);
					return;
				}
				
			}
			
			throw new RuntimeException("A classe "+this.getClass().getName()+" deve declarar tres tipos genricos LISTAGEM, FORMBEAN, BEAN");
		}
		setListagemCommandClass(classes[0]);
		setEntradaCommandClass(classes[1]);
		setBeanClass(classes[2]);
	}
	
	@Override
	public ModelAndView doListagem(WebRequestContext request, FILTRO filtro) throws CrudException {
		try {
			setInfoForTemplate(request);
			setListagemInfo(request, filtro);
			listagem(request, filtro);
		} catch (Exception e) {
			throw new CrudException(LISTAGEM, e);
		}
		return getListagemModelAndView(request, filtro);
	}

	protected void setInfoForTemplate(WebRequestContext request) {
		request.setAttribute("TEMPLATE_listagem", "true");
		request.setAttribute("TEMPLATE_beanNameUncaptalized", getBeanName());
		request.setAttribute("TEMPLATE_beanName", getBeanName());
		request.setAttribute("TEMPLATE_beanDisplayName", Neo.getApplicationContext().getBeanDescriptor(null, beanClass).getDisplayName());
		request.setAttribute("TEMPLATE_beanClass", beanClass);
	}
	
	protected void setInfoForTemplate(WebRequestContext request, FORMBEAN form) {
		request.setAttribute("TEMPLATE_beanNameUncaptalized", getBeanName());
		request.setAttribute("TEMPLATE_beanName", getBeanName());
		request.setAttribute("TEMPLATE_enviar", "salvar");
		request.setAttribute("TEMPLATE_voltar", "listagem");
		request.setAttribute("TEMPLATE_beanDisplayName", Neo.getApplicationContext().getBeanDescriptor(null, beanClass).getDisplayName());
		request.setAttribute("TEMPLATE_beanClass", beanClass);
	}

	protected ModelAndView getListagemModelAndView(WebRequestContext request, FILTRO filtro) {
		return new ModelAndView("crud/"+getBeanName()+"Listagem");
	}

	protected void setListagemInfo(WebRequestContext request, FILTRO filtro) throws Exception {
		if (!listagemVaziaPrimeiraVez() || filtro.isNotFirstTime()) {
			ListagemResult<BEAN> listagemResult = getLista(request, filtro);
			request.setAttribute("lista", listagemResult.list());
			request.setAttribute("currentPage", filtro.getCurrentPage());
			request.setAttribute("numberOfPages", filtro.getNumberOfPages());
			request.setAttribute("filtro", filtro);			
		} else {
			request.setAttribute("lista", new ArrayList());
			request.setAttribute("currentPage", 0);
			request.setAttribute("numberOfPages", 0);
			request.setAttribute("filtro", filtro);
		}
	}

	protected ListagemResult<BEAN> getLista(WebRequestContext request, FILTRO filtro) {
		return genericService.findForListagem(filtro);
	}
	
	protected boolean listagemVaziaPrimeiraVez() {
		return false;
	}

	protected void listagem(WebRequestContext request, FILTRO filtro) throws Exception {
	}

	@Override
	public ModelAndView doEntrada(WebRequestContext request, FORMBEAN form) throws CrudException {
		if("true".equals(request.getParameter("forcarConsulta"))){
			request.setAttribute(CONSULTAR, true);
		}
		try {
			setInfoForTemplate(request, form);
			setEntradaDefaultInfo(request, form);
			entrada(request, form);
		} catch (Exception e) {
			throw new CrudException(ENTRADA, e);
		}
		return getEntradaModelAndView(request, form);
	}



	protected ModelAndView getEntradaModelAndView(WebRequestContext request, FORMBEAN form) {
		if(Boolean.TRUE.equals(request.getAttribute(CONSULTAR))){
			return new ModelAndView("crud/"+getBeanName()+"Consulta");
		} else {
			return new ModelAndView("crud/"+getBeanName()+"Entrada");	
		}
	}

	protected void setEntradaDefaultInfo(WebRequestContext request, FORMBEAN form) throws Exception {
		request.setAttribute(getBeanName(), form);		
	}
	
	protected void entrada(WebRequestContext request, FORMBEAN form) throws Exception {
	}

	@Override
	public ModelAndView doCriar(WebRequestContext request, FORMBEAN form) throws CrudException {
		try {
			BEAN bean = formToBean(form);
			bean = criar(request, bean);
			form = beanToForm(bean);
		} catch (Exception e) {
			throw new CrudException(CRIAR, e);
		}
		return getCriarModelAndView(request, form);
	}

	protected ModelAndView getCriarModelAndView(WebRequestContext request, FORMBEAN form) throws CrudException {
		//return continueOnAction("entrada", form);
		//TODO FAZER O CONTINUETOACTION
		((DefaultWebRequestContext)request).setLastAction("entrada");
		return doEntrada(request, form);
	}

	protected BEAN criar(WebRequestContext request, BEAN form) throws Exception {
		try {
			return beanClass.newInstance();
		} catch (InstantiationException e) {
			throw new NeoException("No foi possvel instanciar classe "+entradaCommandClass, e);
		} catch (IllegalAccessException e) {
			throw new NeoException("No foi possvel instanciar classe "+entradaCommandClass, e);
		}
	}

	@Override
	public ModelAndView doEditar(WebRequestContext request, FORMBEAN form) throws CrudException {
		if(request.getAttribute(CONSULTAR) == null){
			request.setAttribute(CONSULTAR, false);
		}
		if("true".equals(request.getParameter("forcarConsulta"))){
			request.setAttribute(CONSULTAR, true);
		}
		try {
			BEAN bean = formToBean(form);
			bean = carregar(request, bean);
			form = beanToForm(bean);
		} catch (Exception e) {
			throw new CrudException(EDITAR, e);
		}
		return getEditarModelAndView(request, form);
	}

	protected ModelAndView getEditarModelAndView(WebRequestContext request, FORMBEAN form) throws CrudException {
		//TODO FAZER O CONTINUETOACTION
		((DefaultWebRequestContext)request).setLastAction("entrada");
		return doEntrada(request, form);
	}

	protected BEAN carregar(WebRequestContext request, BEAN bean) throws Exception {
		return genericService.loadForEntrada(bean);
	}

	@Override
	public ModelAndView doSalvar(WebRequestContext request, FORMBEAN form) throws CrudException {
		BEAN bean = null;
		try {
			bean = formToBean(form);
			salvar(request, bean);
		} catch (Exception e) {
			throw new CrudException(SALVAR, e);
		}
		return getSalvarModelAndView(request, bean);
	}
	
	protected ModelAndView getSalvarModelAndView(WebRequestContext request, BEAN bean) {
		if("true".equals(request.getParameter("fromInsertOne"))){
			Object id = Util.beans.getId(bean);
			String description = Util.strings.toStringDescription(bean);
			View.getCurrent()
				.println("<html>" +
						"<script language=\"JavaScript\" src=\""+request.getServletRequest().getContextPath()+"/resource/js/util.js\"></script>" +
						"<script language=\"JavaScript\">selecionar('"+id+"', '"+description+"', true);</script>" +
						"</html>");
			return null;
		} else {
			return sendRedirectToAction("listagem");
		}
	}

	protected void salvar(WebRequestContext request, BEAN bean) throws Exception {
		genericService.saveOrUpdate(bean);
	}

	@Override
	public ModelAndView doExcluir(WebRequestContext request, FORMBEAN form) throws CrudException {
		BEAN bean = null; 
		try {
			bean = formToBean(form);
			excluir(request, bean);
		} catch (Exception e) {
			throw new CrudException(EXCLUIR, e);
		}
		return getExcluirModelAndView(request, bean);
	}
	
	protected ModelAndView getExcluirModelAndView(WebRequestContext request, BEAN bean) {
		return sendRedirectToAction("listagem");
	}

	protected void excluir(WebRequestContext request, BEAN bean) throws Exception {
		genericService.delete(bean);
	}

	@SuppressWarnings("unchecked")
	public FORMBEAN beanToForm(BEAN bean){
		if(entradaCommandClass.equals(beanClass)){
			return (FORMBEAN)bean;
		}
		throw new NeoException("No foi possvel converter de bean para form. Sobrescrever o mtodo beanToForm");
	}

	@SuppressWarnings("unchecked")
	public BEAN formToBean(FORMBEAN form){
		if(beanClass.equals(entradaCommandClass)){
			return (BEAN)form;
		}
		throw new NeoException("No foi possvel converter de form para bean. Sobrescrever o mtodo formToBean");
	}
	
	public String getBeanName() {
		return Util.strings.uncaptalize(beanClass.getSimpleName());
	}
	
    protected String[] getAllowedFieldsForSaving(HttpServletRequest request) {
    	//TODO IMPLEMENTAR
    	return null;
    }

}
