/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.classmanager;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import br.com.linkcom.neo.util.ReflectionCache;
import br.com.linkcom.neo.util.ReflectionCacheFactory;


public class ClassManagerImpl implements ClassManager {
	
	private Map<Object, List<Class<?>>> families = new HashMap<Object, List<Class<?>>>();
	private List<Class<?>> classes = new ArrayList<Class<?>>();
	
	//cache
	private Map<Class<?>, List<Class<?>>> classesOfType = new HashMap<Class<?>, List<Class<?>>>();
	private Map<Class<?>, List<Class<?>>> classesOfFamily = new HashMap<Class<?>, List<Class<?>>>();
	private Map<Class<?>, List<Class<?>>> classesWithAnnotation = new HashMap<Class<?>, List<Class<?>>>();
	
	private Map<String, List<Class<?>>> classesFromPackage = new HashMap<String, List<Class<?>>>();

	@SuppressWarnings("unchecked")
	private void fillList(List<Class<?>> list, Class<?> type){
		ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
		for (Class<?> clazz : classes) {
			if(type.isAssignableFrom(clazz) && !list.contains(clazz)){
				list.add(clazz);
			} else if(type.isAnnotation()){
				if(reflectionCache.isAnnotationPresent(clazz, (Class<? extends Annotation>) type)){
					if (!list.contains(clazz)) {
						list.add(clazz);
					}
				}
			}
		}
	}
	
	public Class<?>[] getAllClassesOfType(Class<?> type) {
		List<Class<?>> list = classesOfType.get(type);
		if(list == null){
			list = new ArrayList<Class<?>>();
			fillList(list, type);
		}
		return list.toArray(new Class<?>[list.size()]);
	}

	public Class<?>[] getClassesOfFamily(String family) {
		System.out.println("Get classes of family");
		long x = System.currentTimeMillis();
		List<Class<?>> list = classesOfFamily.get(family);
		if(list == null){
			list = new ArrayList<Class<?>>();
			List<Class<?>> familyClasses = families.get(family);
			for (Class<?> familyClass : familyClasses) {
				fillList(list, familyClass);
			}
		}
		System.out.println(System.currentTimeMillis() - x);
		return list.toArray(new Class<?>[list.size()]);
	}

	public Class<?>[] getClassesWithAnnotation(Class<?> annotationType) {
		List<Class<?>> list = classesWithAnnotation.get(annotationType);
		if(list == null){
			list = new ArrayList<Class<?>>();
			fillList(list, annotationType);
		}
		return list.toArray(new Class<?>[list.size()]);
	}

	public void registerClass(Class<?> clazz) {
		//anotaes nao devem ser registradas
		if(clazz.isAnnotation()){
			return;
		}
		classes.add(clazz);
	}
	

	/**
	 * Anotaes que nao devem ser checadas quanto a familia.
	 * Problema: Recursividade
	 */
//	private Class<?>[] dontEvaluateAnnotations = new Class<?>[]{
//			Documented.class,
//			Target.class,
//			Retention.class
//	};
	
	public void registerFamily(Object family, Class<?>... classes) {
		List<Class<?>> list = families.get(family);
		if(list == null){
			list = new ArrayList<Class<?>>();
			families.put(family, list);
		}
		Collections.addAll(list, classes);
	}

	public Class<?>[] getAllClassesFromPackage(String pacote) {
		List<Class<?>> list = classesFromPackage.get(pacote);
		if(list == null){
			list = new ArrayList<Class<?>>();
			for (Class<?> clazz : classes) {
				if(isInPackage(clazz, pacote)){
					list.add(clazz);
				}
			}
		}
		return list.toArray(new Class<?>[list.size()]);
	}

	private boolean isInPackage(Class<?> clazz, String pacote) {
		return clazz.getName().startsWith(pacote);
	}

	public Class<?>[] getAllClasses() {
		return classes.toArray(new Class<?>[classes.size()]);
	}



}
