/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.bean;

import java.lang.reflect.Method;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.MutablePropertyValues;
import org.springframework.beans.factory.config.RuntimeBeanReference;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.beans.factory.support.RootBeanDefinition;

import br.com.linkcom.neo.bean.annotation.Bean;
import br.com.linkcom.neo.bean.annotation.CrudBean;
import br.com.linkcom.neo.bean.annotation.DAOBean;
import br.com.linkcom.neo.bean.annotation.Ref;
import br.com.linkcom.neo.bean.annotation.ServiceBean;
import br.com.linkcom.neo.persistence.GenericDAO;
import br.com.linkcom.neo.util.ReflectionCache;
import br.com.linkcom.neo.util.ReflectionCacheFactory;
import br.com.linkcom.neo.util.Util;

public class DAOBeanRegister implements TypeBeanRegister {

	protected int autowire = DefaultListableBeanFactory.AUTOWIRE_AUTODETECT;
	
	public Class getBeanClass() {
		return GenericDAO.class;
	}

	public String getName(Class<?> clazz) {
		//se tiver alguma dessas anotacoes o bean deve ser registrado atraves das anotacoes
		if(clazz.getAnnotation(Bean.class) != null
				|| clazz.getAnnotation(CrudBean.class) != null
				|| clazz.getAnnotation(ServiceBean.class) != null
				|| clazz.getAnnotation(DAOBean.class) != null){
			return null;
		}
		return Util.strings.uncaptalize(clazz.getSimpleName());
	}

	public void registerBeans(List<Class<?>> classes, DefaultListableBeanFactory beanFactory) {
		for (Class<?> class1 : classes) {
			String name = getName(class1);
			if(name != null){
				MutablePropertyValues mutablePropertyValues = createMutablePropertyValues(class1);
				RootBeanDefinition beanDefinition = createBeanDefinition(class1, mutablePropertyValues);
				beanDefinition.setAutowireMode(autowire);
				beanFactory.registerBeanDefinition(name, beanDefinition);
			}
		}
	}

	protected MutablePropertyValues createMutablePropertyValues(Class<?> class1) {
		MutablePropertyValues mutablePropertyValues = new MutablePropertyValues();
//		registrar os refs 
		ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
		Method[] methods = reflectionCache.getMethods(class1);
		for (Method method : methods) {
			if(method.getName().startsWith("set")){
				String propertyName = StringUtils.uncapitalize(method.getName().substring(3));
				Ref refAnnotation = method.getAnnotation(Ref.class);
				if (refAnnotation != null) {
					String name = null;
					if (!StringUtils.isEmpty(refAnnotation.bean())) {
						name = refAnnotation.bean();
					}
					if("<null>".equals(name)){
						mutablePropertyValues.addPropertyValue(propertyName, null);
					} else {
						if(autowire != 0){
							if(name != null){
								// se estiver no modo autowire s colocar o propertyvalue se tiver sido explicito
								mutablePropertyValues.addPropertyValue(propertyName, new RuntimeBeanReference(name));
							}
						} else {
							if(name == null){
								name = propertyName; 
							}
							mutablePropertyValues.addPropertyValue(propertyName, new RuntimeBeanReference(name));
						}	
					}
					
											
				}
			}
		}
		return mutablePropertyValues;
	}

	protected RootBeanDefinition createBeanDefinition(Class<?> class1, MutablePropertyValues mutablePropertyValues) {
		RootBeanDefinition beanDefinition = new RootBeanDefinition(class1, mutablePropertyValues);
		return beanDefinition;
	}
}
