/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.bean;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.MutablePropertyValues;
import org.springframework.beans.factory.config.RuntimeBeanReference;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.beans.factory.support.RootBeanDefinition;

import br.com.linkcom.neo.bean.annotation.Bean;
import br.com.linkcom.neo.bean.annotation.Property;
import br.com.linkcom.neo.bean.annotation.Ref;
import br.com.linkcom.neo.util.ReflectionCache;
import br.com.linkcom.neo.util.ReflectionCacheFactory;

/**
 * Registra beans com anotao @Bean
 * @author rogelgarcia
 * @since 20/10/2005
 * @version 1.0
 */
public class BeanRegisterBean implements br.com.linkcom.neo.bean.AnnotatedBeanRegister {
	
	protected int autowire = DefaultListableBeanFactory.AUTOWIRE_AUTODETECT;
	
	protected Log log = LogFactory.getLog("BeanRegister");

	public Class<? extends Annotation> getAnnotationClass() {
		return Bean.class;
	}

	public void registerBeans(List<Class<?>> annotatedClasses, DefaultListableBeanFactory beanFactory) {
		for (Class<?> annotatedClass : annotatedClasses) {
			log.debug("Registrando bean automaticamente ... "+annotatedClass.getName());
			registerBean(beanFactory, annotatedClass);
		}
	}

	protected void registerBean(DefaultListableBeanFactory beanFactory, Class<?> class1) {
		ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
		Bean bean = reflectionCache.getAnnotation(class1, Bean.class);
		registerBean(beanFactory, class1, bean);
	}
	
	protected void registerBean(DefaultListableBeanFactory beanFactory, Class<?> class1, Bean bean) {
		MutablePropertyValues mutablePropertyValues = createPropertyValues(class1, bean);
		RootBeanDefinition beanDefinition = createBeanDefinition(class1, mutablePropertyValues);
		beanDefinition.setAutowireMode(autowire);
		String beanName = getBeanName(class1, bean);
		beanFactory.registerBeanDefinition(beanName, beanDefinition);
	}
	
	protected RootBeanDefinition createBeanDefinition(Class<?> class1, MutablePropertyValues mutablePropertyValues) {
		RootBeanDefinition beanDefinition = new RootBeanDefinition(class1, mutablePropertyValues);
		return beanDefinition;
	}
	
	protected MutablePropertyValues createPropertyValues(Class<?> class1, Bean bean) {
		MutablePropertyValues mutablePropertyValues = new MutablePropertyValues();
		//registrar os beanProperties
		Property[] beanProperties = bean.properties();
		for (Property property : beanProperties) {
			Object propvalue = null;
			String value = property.value();
			if(!StringUtils.isEmpty(value)){
				if(value.equals("null")){
					propvalue = null;
				} else {
					propvalue = value;						
				}
			} else if(!StringUtils.isEmpty(property.ref())){
				if(property.ref().equals("null")){
					propvalue = null;
				} else {
					propvalue = new RuntimeBeanReference(property.ref());						
				}
			} else {
				throw new RuntimeException("A classe "+class1.getName()+" no declarou o atributo beanProperties corretamente, ref ou value devem ser informados em cada beanProperty. Erro na propriedade: "+property.name());
			} 
			if(!StringUtils.isEmpty(value)&&!StringUtils.isEmpty(property.ref())){
				throw new RuntimeException("A classe "+class1.getName()+" no declarou o atributo beanProperties corretamente, somente ref ou value deve ser informado em cada beanProperty. Erro na propriedade: "+property.name());
			}
			
			mutablePropertyValues.addPropertyValue(property.name(), propvalue);			
		}
		//registrar os refs 
		ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
		Method[] methods = reflectionCache.getMethods(class1);
		for (Method method : methods) {
			if(method.getName().startsWith("set")){
				String propertyName = StringUtils.uncapitalize(method.getName().substring(3));
				Ref refAnnotation = method.getAnnotation(Ref.class);
				if (refAnnotation != null) {
					String name = null;
					if (!StringUtils.isEmpty(refAnnotation.bean())) {
						name = refAnnotation.bean();
					}
					if("<null>".equals(name)){
						mutablePropertyValues.addPropertyValue(propertyName, null);
					} else {
						if(autowire != 0){
							if(name != null){
								// se estiver no modo autowire s colocar o propertyvalue se tiver sido explicito
								mutablePropertyValues.addPropertyValue(propertyName, new RuntimeBeanReference(name));
							}
						} else {
							if(name == null){
								name = propertyName; 
							}
							mutablePropertyValues.addPropertyValue(propertyName, new RuntimeBeanReference(name));
						}	
					}
					
											
				}
			}
		}
		return mutablePropertyValues;
	}
	
	public String getBeanName(Class<?> class1, Bean bean) {
		String beanName = bean.name();
		if(StringUtils.isEmpty(beanName)){
			beanName = StringUtils.uncapitalize(class1.getSimpleName());
		}
		return beanName;
	}

	public String getName(Class<?> clazz) {
		ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
		Bean bean = reflectionCache.getAnnotation(clazz, Bean.class);
		return getBeanName(clazz, bean);
	}
}
