/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.bean;

import java.lang.reflect.Method;

import javax.persistence.Id;

import br.com.linkcom.neo.bean.annotation.DescriptionProperty;
import br.com.linkcom.neo.exception.BeanDescriptorCreationException;
import br.com.linkcom.neo.exception.NeoException;
import br.com.linkcom.neo.util.ReflectionCache;
import br.com.linkcom.neo.util.ReflectionCacheFactory;
import br.com.linkcom.neo.util.Util;

/**
 * @author rogelgarcia
 * @since 23/01/2006
 * @version 1.1
 */
public class BeanDescriptorImpl<E> implements BeanDescriptor<E> {
	
	private E bean;
	private Class<E> clazz;
	private String descriptionPropertyName;
	private String idPropertyName;
	private String displayName;
	private PropertyDescriptorBeanWrapper beanWrapper;
	
	public PropertyDescriptorBeanWrapper getBeanWrapper() {
		return beanWrapper;
	}

	@SuppressWarnings("unchecked")
	public BeanDescriptorImpl(E bean, Class<E> clazz) throws NullPointerException, IllegalArgumentException {
		if(bean == null && clazz == null){
			throw new BeanDescriptorCreationException("Erro ao criar BeanDescriptor. O objeto  nulo e a classe no foi informada");
		}
		this.bean = bean;
		if(clazz != null && bean != null && !clazz.isAssignableFrom(bean.getClass())){
			throw new IllegalArgumentException("Erro ao criar BeanDescriptor. O objeto passado no  do tipo da classe informada");
		}
		this.clazz = clazz == null? (Class<E>)bean.getClass() : clazz;
		if(this.clazz.getName().contains("$$")){
			Class superclass = this.clazz.getSuperclass();
			this.clazz = superclass; // ignorar as classes do hibernate
		}
		beanWrapper = new PropertyDescriptorBeanWrapper(this.clazz, bean);
		beanWrapper.setIndexValueResolver(new IndexValueResolverImpl());
	}

	public String getDescriptionPropertyName() {
		ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
		if(descriptionPropertyName == null){
			Class clazz = this.clazz;
			while (!clazz.equals(Object.class) && descriptionPropertyName == null) {
				Method[] methods = reflectionCache.getMethods(clazz);
				for (Method method : methods) {
					if (reflectionCache.isAnnotationPresent(method, DescriptionProperty.class)) {
						if (descriptionPropertyName != null) {
							throw new NeoException("A classe " + clazz + " declarou 2 @DescriptionProperty");
						}
						descriptionPropertyName = Util.beans.getPropertyFromGetter(method.getName());
					}
				}
				clazz = clazz.getSuperclass();
			}
		}
		return descriptionPropertyName;
	}

	public String getIdPropertyName() {
		ReflectionCache reflectionCache = ReflectionCacheFactory.getReflectionCache();
		if(idPropertyName == null){
			Class clazz = this.clazz;
			while (!clazz.equals(Object.class) && idPropertyName == null) {
				Method[] methods = reflectionCache.getMethods(clazz);
				for (Method method : methods) {
					if (reflectionCache.isAnnotationPresent(method, Id.class)) {
						if (idPropertyName != null) {
							throw new NeoException("A classe " + clazz + " declarou 2 @Id");
						}
						idPropertyName = Util.beans.getPropertyFromGetter(method.getName());
					}
				}
				clazz = clazz.getSuperclass();
			}
		}
		return idPropertyName;
	}

	public Object getDescription() {
		if(getDescriptionPropertyName() != null){
			return getPropertyDescriptor(getDescriptionPropertyName()).getValue();
		}
		return null;
	}

	public Object getId() {
		if(getIdPropertyName() != null){
			return getPropertyDescriptor(getIdPropertyName()).getValue();
		}
		return null;
	}

	public String getDisplayName() {
		if (displayName == null) {
			displayName = DisplayNameUtil.getDisplayName(clazz);
		}
		return displayName;
	}

	public E getTargetBean() {
		return bean;
	}

	public Class<E> getTargetClass() {
		return clazz;
	}

	public PropertyDescriptor getPropertyDescriptor(String propertyName) {
		return new PropertyDescriptorImpl(this, propertyName);
	}

	public void setIndexValueResolver(IndexValueResolver indexValueResolver) {
		beanWrapper.setIndexValueResolver(indexValueResolver);
	}

}
