/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
/*
 * Criado em 21/03/2005
 *
 */
package br.com.linkcom.neo.authorization.impl;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import br.com.linkcom.neo.authorization.AuthorizationDAO;
import br.com.linkcom.neo.authorization.AuthorizationItem;
import br.com.linkcom.neo.authorization.AuthorizationModule;
import br.com.linkcom.neo.authorization.Permission;
import br.com.linkcom.neo.authorization.PermissionLocator;
import br.com.linkcom.neo.authorization.Role;
import br.com.linkcom.neo.authorization.User;
import br.com.linkcom.neo.controller.ControlMapping;
import br.com.linkcom.neo.core.standard.RequestContext;


/**
 * Implementacao da interface para localizar as Permissions
 * @author rogelgarcia
 */
public class PermissionLocatorImpl implements PermissionLocator {
	
	static Log log = LogFactory.getLog(PermissionLocator.class);

	private static final String CACHE_ROLES = "CACHE_ROLES";
	
	protected AuthorizationDAO authorizationDAO;
	
	public void setAuthorizationDAO(AuthorizationDAO authorizationDAO) {
		this.authorizationDAO = authorizationDAO;
	}

    Map<ControlMapping, Map<Role, Permission>> cache = new HashMap<ControlMapping, Map<Role, Permission>>();
     
    
    public synchronized Permission[] getPermissions(RequestContext request, User user, ControlMapping control) {
    	Map<User, Role[]> cacheRoles = getCacheRoles(request);    
        //tentar achar roles no cache
    	Role[] userRoles = cacheRoles.get(user);
		if (userRoles == null) {
			// buscar do banco
			userRoles = authorizationDAO.findUserRoles(user);
			cacheRoles.put(user, userRoles);
		}

        Permission[] permissions = new Permission[userRoles.length];
        for (int i = 0; i < userRoles.length; i++) {
        	String controlName = control.getName();
        	Role role = userRoles[i];
        	
            //tentar cache
        	Map<Role, Permission> mapRolePermission;
			mapRolePermission = cache.get(control);
        	if(mapRolePermission!=null){
        		Permission permission = mapRolePermission.get(role);
        		if(permission != null){ //achou no cache, pular o resto do loop
        			log.debug("Using cached permission: "+permission);
        			permissions[i] = permission;
        			continue;
        		}
        	} else {
        		//criar o mapa, mais tarde nesse loop ser populado
        		mapRolePermission = new HashMap<Role, Permission>();
				cache.put(control, mapRolePermission);

        	}
            
            //tentar banco
			permissions[i] = authorizationDAO.findPermission(role, controlName);
            
            if(permissions[i]==null){
            	//criar mapa de autorizacao default
            	AuthorizationModule authorizationModule = control.getAuthorizationModule();
				Map<String, String> defaultPermissionMap = getDefaultPermissionMap(authorizationModule);
            	//adicionar permissao faltante no banco
            	log.debug("Criando permissao... control="+controlName+", role="+role.getName());
            	permissions[i] = authorizationDAO.savePermission(controlName, role, defaultPermissionMap);
                /*
            	throw new InconsistencyException("Inconsistencia de dados: " +
                		"No h permisso para o papel '"+role.getName()+"' para acessar o Controller '"+control.getName()+"'");
                */
            }
            
			cache.get(control).put(role, permissions[i]);

            	
        }
        return permissions;
    }


	@SuppressWarnings("unchecked")
	private Map<User, Role[]> getCacheRoles(RequestContext request) {
		Map<User, Role[]> cache = (Map<User, Role[]>) request.getUserAttribute(CACHE_ROLES);
		if(cache == null){
			cache = new HashMap<User, Role[]>();
			request.setUserAttribute(CACHE_ROLES, cache);
		}
		return cache;
	}


	private Map<String, String> getDefaultPermissionMap(AuthorizationModule authorizationModule) {
		AuthorizationItem[] authorizationItens = authorizationModule.getAuthorizationItens();
		Map<String, String> defaultPermissionMap = new HashMap<String, String>();
		
		for (AuthorizationItem item : authorizationItens) {
			String id = item.getId();
			if(item.getValores()== null || item.getValores().length == 0) throw new IllegalArgumentException("Os valores possveis de um item de autorizao no pode ser um array vazio ou null");
			String valorMaisRestritivo = item.getValores()[item.getValores().length-1];
			defaultPermissionMap.put(id, valorMaisRestritivo);
		}
		return defaultPermissionMap;
	}


    public synchronized void reset() {
    	cache = new HashMap<ControlMapping, Map<Role, Permission>>();
    }


	public void clearCache() {
		reset();
	}

}
