/*
 * Neo Framework http://www.neoframework.org
 * Copyright (C) 2007 the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * You may obtain a copy of the license at
 * 
 *     http://www.gnu.org/copyleft/lesser.html
 * 
 */
package br.com.linkcom.neo.authorization.impl;

import br.com.linkcom.neo.authorization.Authorization;
import br.com.linkcom.neo.authorization.AuthorizationManager;
import br.com.linkcom.neo.authorization.AuthorizationModule;
import br.com.linkcom.neo.authorization.HasAccessAuthorizationModule;
import br.com.linkcom.neo.authorization.Permission;
import br.com.linkcom.neo.authorization.PermissionLocator;
import br.com.linkcom.neo.authorization.User;
import br.com.linkcom.neo.authorization.UserLocator;
import br.com.linkcom.neo.controller.ControlMapping;
import br.com.linkcom.neo.controller.ControlMappingLocator;
import br.com.linkcom.neo.controller.MultiActionController;
import br.com.linkcom.neo.core.standard.Neo;
import br.com.linkcom.neo.core.standard.RequestContext;
import br.com.linkcom.neo.core.web.WebRequestContext;

/**
 * GERENCIADOR DE AUTORIZAO
 * Contm mtodos para verificar authorizao, permissao, etc. dentro da aplicao
 * Deve ser adquirido atravs do FrameworkContext
 */  
public class AuthorizationManagerImpl implements AuthorizationManager{

	public static final String ACTION_PARAMETER = MultiActionController.ACTION_PARAMETER;
	
	protected UserLocator userLocator;
	protected PermissionLocator permissionLocator;
	protected ControlMappingLocator controlMappingLocator;
	

	public void setControlMappingLocator(ControlMappingLocator controlMappingLocator) {
		this.controlMappingLocator = controlMappingLocator;
	}

	public void setPermissionLocator(PermissionLocator permissionLocator) {
		this.permissionLocator = permissionLocator;
	}

	public void setUserLocator(UserLocator userLocator) {
		this.userLocator = userLocator;
	}
	
	/**
	 * Verifica se o usurio tem permissao para esse request e salva um atributo na requisicao
	 * representando a autorizacao
	 * @param httpServletRequest
	 * @return
	 */
	public boolean isAutorized(RequestContext request) {
		String actionParameter = request.getParameter(ACTION_PARAMETER);
	
		User usuario = userLocator.getUser(request);

		String path = request.getRequestQuery();
		ControlMapping controlMapping = controlMappingLocator.getControlMapping(path);

		return isAuthorized(request, actionParameter, usuario, permissionLocator, controlMapping);
	}

	public boolean isAuthorized(RequestContext request, String actionParameter, User usuario, PermissionLocator permissionLocator, ControlMapping controlMapping) {
		boolean isAuthorized;
		if(controlMapping==null|| controlMapping.getAuthorizationModule()==null || controlMapping.getAuthorizationModule() instanceof HasAccessAuthorizationModule){
            isAuthorized = true;
        }
		else {
            Authorization autorizacao = createAuthorization(request, usuario, permissionLocator, controlMapping);

			AuthorizationModule moduloAutorizacao = controlMapping.getAuthorizationModule();

			if (moduloAutorizacao.isAuthorized(actionParameter, autorizacao)) {
                isAuthorized = true;
            } else {
                isAuthorized = false;
            }
			//salva a autorizacao na requisicao
			request.setAttribute(Authorization.AUTHORIZATION_ATTRIBUTE, autorizacao);
        }
		return isAuthorized;
	}
	
	/**
	 * Verifica se o usurio tem autorizacao para determinado path e actionparameter
	 * @param path path do controller 
	 * @param actionParameter parametro (ex.: CRIAR, relativo a cada tipo de controler)
	 * @param httpServletRequest requisicao atual (busca o usurio atual da requisicao)
	 * @return
	 */
	public boolean isAuthorized(String path, String actionParameter, RequestContext request){
		User user = userLocator.getUser(request);
		return isAuthorized(path, actionParameter, user);
	}
	
	public boolean isAuthorized(String path, String actionParameter, User usuario){
		// pega o localizadorControl no contexto do framework		
		ControlMapping controlMapping = controlMappingLocator.getControlMapping(path);
		return isAuthorized(Neo.getRequestContext(), actionParameter, usuario, permissionLocator, controlMapping);
	}
	/*
	public boolean isAuthorized(String actionParameter, User usuario, PermissionLocator permissionLocator, ControlMapping controlMapping) {
		boolean isAuthorized;
		if(controlMapping==null|| controlMapping.getAuthorizationModule()==null || controlMapping.getAuthorizationModule() instanceof HasAccessAuthorizationModule){
            isAuthorized = true;
        }
		else {
            Authorization autorizacao = createAuthorization(usuario, permissionLocator, controlMapping);

			AuthorizationModule moduloAutorizacao = controlMapping.getAuthorizationModule();

			if (moduloAutorizacao.isAuthorized(actionParameter, autorizacao)) {
                isAuthorized = true;
            } else {
                isAuthorized = false;
            }
        }
		return isAuthorized;
	}
*/
	private Authorization createAuthorization(RequestContext request, User usuario, PermissionLocator permissionLocator, ControlMapping controlMapping) {
		Permission[] permissoes;
		if(usuario!=null){
		    permissoes = permissionLocator.getPermissions(request, usuario, controlMapping);
		} else {
		    permissoes = new Permission[0];
		}
		Authorization autorizacao = (Authorization) controlMapping.getAuthorizationModule().createAuthorization(permissoes);
		return autorizacao;
	}

	public void loadPermissions(WebRequestContext requestContext) {
		userLocator.loadPermissions(requestContext);
	}
	
}
